#########################################################################################
# st_mail.pm : This file includes the mail routine:					#
#											#
#	       mail_routine ($dbh, fault_id, action, \@users_add_del, \@users)		#
#											#
#	       - $dbh 			: For database connection			#
#	       - fault_id or file_id 	: The fault_id for the email process		#
#	       - action 	 	: The action (action fault)			#
#	       - \@users_add_del 	: A table (array) with users_ids (for emails)	#
#					  >>> case of action 7 & 8 only.		#	
#	       - \@users 	 	: A table (array) with users_ids (for emails)	#	
#											#
# Artists : 	Politis Konstantinos							#
# 		Theodore J. Soldatos							#
#											#
# Copyright (C) 2004-2005 Space Hellas							#
# Copyright (C) 2004-2005 Theodore J. Soldatos						#
#											#
# This program is free software; you can redistribute it and/or				#
# modify it under the terms of the GNU General Public License				#
# as published by the Free Software Foundation; either version 2			#
# of the License, or (at your option) any later version.				#
#											#
# This program is distributed in the hope that it will be useful,			#
# but WITHOUT ANY WARRANTY; without even the implied warranty of			#
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the				#
# GNU General Public License for more details.						#
#											#
# You should have received a copy of the GNU General Public License			#
# along with this program; if not, write to the Free Software				#
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,				#
# MA  02110-1301, USA									#
#											#
# History										#
# -------										#
# Version 0.1 : Startup Version 							#
#               Support the Basic Mail Mechanism					#
# Version 0.2 : Fix parameter $dbh for database connection				#
#		Fix <subject> and <body text> of mails					#
#		-----									#
#		Note:									#
#		-----									#
#		Depending on the <action> flag the above two (2)			#
#		parameters are building into the subroutines				#
#		<action>								#
#		1: New entry in st_faultsmsglog						#
#		2: New entry in st_faultslevellog					#
#		3: New entry in st_faultsstatuslog					#
#		4: New entry in st_faultstechs						#
#		5: Fault closed								#
#		6: Fault opened								#		
#		7: Related users assignment						#
#		8: Related users removal						#
#		9: New File Upload							#
#											#
# Version 0.3 : Fix mail format (subject & body text)					#
# Version 0.4 : Fix encoding greek subject and greek body text 				#
# ------------------------------------------------------------->>>>>			#
# Version 1.0 : Fix operation, mail format and greek support				#
# ------------------------------------------------------------>>>>>>>			#
#											#
# Version 2.0 : Fix at last the bug with the greek encoding (ISO-8859-7)		#
#		of ('big size' more than 30 characters) subject.			#
# Version 2.1 : Start fixing the new actions (6-8).					#
# Version 2.2 : Continue fixing new actions (6-8) [possible changes to the future]	#
#		This is the 1st version of these actions (testing and fixing).		#
# Version 2.3 : Fix bugs - email routine - now supports the follow concepts:		#
#		- If the same user appears twice in a fault, only one mail should 	#
#	  	  be sent.								#
#		- The full name of the user must appear in To:				#
#		- Use st_lib::datetostr to print dates in mails.			#
# Version 2.3.1 : Fix bugs 								#
# Version 2.3.2 : Make SMTP server and From address in st_mail parameters from st_lib.	#
# Version 2.3.3 : Added footer with link to fault. Translated everything (?) to engl.	#
# Version 2.3.4 : Fix bugs:								#
#		  1. Fix the subject length - cut the first 50 characters,		#
#		  2. Fix "Fault description" on body text - action: Fault opened.	# 
# Version 2.3.5 : Terminology review and update						#
# Version 2.3.6 : Fix bug survey links in closing mails. Appear now only if the		#
#		  specific ticket type *has* a survey.					#
# Version 2.3.7 : Fix problem in related users mail.					#
# Version 2.3.8 : Add another action (9 - File upload).	Sent mails now			#
#		  when a new file uploaded (see st_files)				#
# Version 2.3.9 : New log mails now use short descr as subject. Removed 50 chars limit.	#
# Version 2.4.0 : Added check_mailprefs routine and calls to it.			#
# Version 2.4.1 : Added action 10 (survey answered)					#
# Version 2.4.2 : Replaced hardcoded encodings with parameters.				#
#											#
#########################################################################################

package st_mail;
use strict;
use st_lib;
use DBI;
use Mail::Sender;				# Perl Module for sending mails
use MIME::QuotedPrint;				# Perl Module for encoding - or decoding - of quoted-printable strings 

my ($sth, $sth1); 						# SQL statement handler. 
my ($rc, $strrc);						# Errors.
my ($dst, $dst1);						# used for dynamic statements.
my ($rest, $tmp, $scrapret, @row_ary);				# Various tmp variables.
my ($subject, $text_body, $tmp_part);				# Variables for returning the subject and the text body of mail.
my (@temp_email_1st, @temp_email_2nd, @temp_email_3rd);		# Temp variables for email (names) accounts of related users.
my ($fullname, $to_mail);

### From st_lib ###

my $smtpserver = $st_lib::smtpserver;
my $fromaddress = $st_lib::fromaddress;
my $url = $st_lib::url;
my $url2 = $st_lib::url2;
my $sencoding = $st_lib::sencoding;

### The <mail_routine> for mail support (for user faults) ###

sub mail_routine 
{
my ($dbh, $fault_id, $action, $users_add_del, $users) = @_;
	
	### Check the action please ###
	### Depending on the action, the subject and the <body> text mail must be built ###

	if ($action eq 1)	
	{
		### New entry in st_faultsmslog (the text should be included) ###

		($subject,$text_body) = case_sub_st_faultsmsglog ($dbh, $fault_id);	
	}
	elsif ($action eq 2)
	{
		### New entry in st_faultslevellog ###
		
		($subject,$text_body) = case_sub_st_faultslevellog ($dbh, $fault_id);
	}
	elsif ($action eq 3)
	{
		### New entry in st_faultstatuslog ###

		($subject,$text_body) = case_sub_st_faultsstatuslog ($dbh, $fault_id);
	}
	elsif ($action eq 4)
	{
		### New entry in st_faultstechs ###

		($subject,$text_body) = case_sub_st_faultstechs ($dbh, $fault_id);
	}
	elsif ($action eq 5)
	{
		### Fault closed ###
	
		($subject,$text_body) = case_sub_st_faultclosed ($dbh, $fault_id);
	}
	elsif ($action eq 6)
	{
		### Opening a fault ###
	
		($subject,$text_body) = case_sub_st_faultopened ($dbh, $fault_id);
	}
	elsif ($action eq 7)
	{
		### Related users assignment ###
	
		($subject,$text_body) = case_sub_st_fault_rel_assignment ($dbh, $fault_id, $$users_add_del[0]);
	}
	elsif ($action eq 8)
	{
		### Related users removal ###
	
		($subject,$text_body) = case_sub_st_fault_rel_removal ($dbh, $fault_id, $users_add_del);
	}
	elsif ($action eq 9)
	{
		### File upload ###
	
		#-- At this case fault_id is actually the file_id on st_files file --#
		
		($subject,$text_body) = case_sub_st_file_upload ($dbh, $fault_id);
	}
	elsif ($action eq 10)
	{
		### Survey answered ###
	
		#-- At this case fault_id is actually the file_id on st_files file --#
		
		($subject,$text_body) = case_sub_st_surv_answered ($dbh, $fault_id);
	}
	else
	{
		print STDERR "Wrong Action <1-9>\n";
	}
	
	###--------------------------------------------------------------------------###
	#-- Case of action (9, file upload) find the correct fault_id from "file_id" --#
	###--------------------------------------------------------------------------###
	
	if ($action eq 9)
	{
		
		$dst = "SELECT u.fault_id
			FROM st_files u WHERE u.id = $fault_id;";
        
		$sth = $dbh->prepare($dst);
		$sth->execute;

		if (st_lib::sthErr($sth->err, $sth->errstr)) { 
			# Failed 
			print "Failed !!!";
			$dbh->disconnect;
			exit;
		};
		
		if (@row_ary = $sth->fetchrow_array) 
		{	$fault_id =  $row_ary[0];	}
		
		$sth->finish;
	}

	
	### Cut the first 50 characters from subject ###
	# No need for that anymore, log entries do not appear in subject.
	#$tmp_part = substr($subject,101);	
	#$subject = substr($subject,0,100);
	#if ($tmp_part ne '') {
	#	$subject = $subject . "...(more)";
	#};
	
	### Encoding $subject - Greek support ###
	$subject = encode_qp($subject);
	$subject =~ s/=\r?\n//g;	# Remove the new line added from encode_qp function

	# Footer to all mails.
	$text_body = $text_body . "\n\nThe follow link points to this Service Request: ";
	$text_body = $text_body . "\n\n$url2+" . $fault_id;
	$text_body = $text_body . "\n\n";
	$text_body = $text_body . "\n+----------------------------------------------------------+";
	$text_body = $text_body . "\n!This is an automatically sent e-mail. Please do not reply.!";
	$text_body = $text_body . "\n!          Thank you for using Space Ticketing NT          !";
	$text_body = $text_body . "\n+----------------------------------------------------------+";
	
	### Start checking cases...	###
	if (($$users[0] eq '') or ($action eq 7) or ($action eq 8) or ($action eq 9)) 
	{
		### No email table is defined so check the st_faults_users_rel, st_faults (logged_by_user_id), 
		### st_tech_of_fault (Table from view).
		
		### Check the st_faults_users_rel at first ###
		
		if ($action eq 7)
		{
			$dst = "SELECT u.user_id 
				FROM st_faults_users_rel u WHERE u.fault_id = $fault_id AND u.user_id != $$users_add_del[0];";
		}
		else
		{
			$dst = "SELECT u.user_id 
				FROM st_faults_users_rel u WHERE u.fault_id = $fault_id;";
		}
		
		$sth = $dbh->prepare($dst);
		$sth->execute;
		
		if (st_lib::sthErr($sth->err, $sth->errstr)) { 
			# Failed 
			print "Failed !!!";
			st_lib::wDbg("ERROR: SELECT 1 Failed ", "st_mail");
			$dbh->disconnect;
			exit;
		};
		
		while (@row_ary = $sth->fetchrow_array) 
		{
			if (check_mailprefs($dbh, $row_ary[0], $action)) {
				# Send mail only if enabled by the user.
				### Based on user_id, find the mail (account) of the current user ###
	
				$dst1 = "SELECT u.email, u.fullname  
				         FROM st_users u WHERE u.id = $row_ary[0] AND u.active = TRUE";
       	 
				$sth1 = $dbh->prepare($dst1);
				$sth1->execute;
	
				if (st_lib::sthErr($sth1->err, $sth1->errstr)) { 
					# Failed 
					print "Failed !!!";
					st_lib::wDbg("ERROR: SELECT 2 Failed ", "st_mail");
					$dbh->disconnect;
					exit;
				};
					
				if (@temp_email_1st = $sth1->fetchrow_array) 
				{
					### This is a valid-proper mail (account) ---> ###
					### Send mail to the current mail account ###
					
					$to_mail = "$temp_email_1st[1] <$temp_email_1st[0]>";
					$to_mail = encode_qp($to_mail);
					$to_mail =~ s/=\r?\n//g;	# Remove the new line added from encode_qp function
				
					#print "Mail Address = $to_mail\n";
	
					(new Mail::Sender)->MailMsg( {smtp => $smtpserver,
								      from => $fromaddress,
								      to => '=?' . $sencoding . '?Q?' . $to_mail . '?=',
								      subject => '=?' . $sencoding . '?Q?' . $subject . '?=',
								      msg => $text_body,
							      	      charset => $sencoding} );
				};
				$sth1->finish;
			}; # if (check_mailprefs($row_ary[0], $action)
		}
		$sth->finish;

		### Check the st_faults (logged_by_user_id) ###
		
		if ($action eq 7)
		{
			$dst = "SELECT u.logged_by_user_id 
				FROM st_faults u WHERE $fault_id = u.id AND u.logged_by_user_id != $$users_add_del[0];";
		}
		else
		{
			$dst = "SELECT u.logged_by_user_id 
				FROM st_faults u WHERE $fault_id = u.id;";
		}
		        
		$sth = $dbh->prepare($dst);
		$sth->execute;

		if (st_lib::sthErr($sth->err, $sth->errstr)) { 
			# Failed 
			print "Failed !!!";
			st_lib::wDbg("ERROR: SELECT 3 Failed ", "st_mail");
			$dbh->disconnect;
			exit;
		};
#		my $dbgvar = $sth->rows;	

		if (@row_ary = $sth->fetchrow_array) {
			if (check_mailprefs($dbh, $row_ary[0], $action)) {
				# Send mail only if enabled by the user.
				### Based on logged_by_user_id, find the mail (account) of the current user ###
	
				$dst1 = "SELECT u.email, u.fullname
				         FROM st_users u WHERE u.id = $row_ary[0] AND u.active = TRUE";
       	 
				$sth1 = $dbh->prepare($dst1);
				$sth1->execute;
	
				if (st_lib::sthErr($sth1->err, $sth1->errstr)) { 
					# Failed 
					print "Failed !!!";
					st_lib::wDbg("ERROR: SELECT 4 Failed ", "st_mail");
					$dbh->disconnect;
					exit;
				}
					
				if (@temp_email_2nd = $sth1->fetchrow_array) 
				{
					### This is a valid-proper mail (account) ---> ###
					### Send mail to the current mail account ###
					
					if (!($temp_email_1st[0] eq $temp_email_2nd[0]))
					{
						$to_mail = "$temp_email_2nd[1] <$temp_email_2nd[0]>";
						$to_mail = encode_qp($to_mail);
						$to_mail =~ s/=\r?\n//g;	# Remove the new line added from encode_qp function
	
						#print "Mail Address = $to_mail\n";
						
						(new Mail::Sender)->MailMsg( {smtp => $smtpserver,
									      from => $fromaddress,
									      to => '=?' . $sencoding . '?Q?'.$to_mail.'?=',
									      subject => '=?' . $sencoding . '?Q?'.$subject.'?=',
									      msg => $text_body,
							      	      	      charset => $sencoding} );
					};
				};
				$sth1->finish;
			}; # if (check_mailprefs($row_ary[0], $action)
		}
		$sth->finish;
		
		### Check the st_tech_of_fault (Table from view) ###
		
		if ($action eq 7)
		{
			$dst = "SELECT u.assigned_to_user_id 
			 	FROM st_tech_of_fault u WHERE $fault_id = u.id AND u.assigned_to_user_id != $$users_add_del[0];";
		}
		else
		{
			$dst = "SELECT u.assigned_to_user_id 
				FROM st_tech_of_fault u WHERE $fault_id = u.id;";
		}

		$sth = $dbh->prepare($dst);
		$sth->execute;

		if (st_lib::sthErr($sth->err, $sth->errstr)) { 
			# Failed 
			print "Failed !!!";
			st_lib::wDbg("ERROR: SELECT 5 Failed ", "st_mail");
			$dbh->disconnect;
			exit;
		};
#		my $dbgvar = $sth->rows;	
		if (@row_ary = $sth->fetchrow_array) 
		{
			if (check_mailprefs($dbh, $row_ary[0], $action)) {
				# Send mail only if enabled by the user.
				### Based on logged_by_user_id, find the mail (account) of the current user ###
	
				$dst1 = "SELECT u.email, u.fullname
				         FROM st_users u WHERE u.id = $row_ary[0] AND u.active = TRUE";
       	 
				$sth1 = $dbh->prepare($dst1);
				$sth1->execute;
	
				if (st_lib::sthErr($sth1->err, $sth1->errstr)) { 
					# Failed 
					print "Failed !!!";
					st_lib::wDbg("ERROR: SELECT 6 Failed ", "st_mail");
					$dbh->disconnect;
					exit;
				}
					
				if (@temp_email_3rd = $sth1->fetchrow_array) 
				{
					### This is a valid-proper mail (account) ---> ###
					### Send mail to the current mail account ###
					
					if ((!($temp_email_3rd[0] eq $temp_email_2nd[0])) and (!($temp_email_3rd[0] eq $temp_email_1st[0])))
					{	
						$to_mail = "$temp_email_3rd[1] <$temp_email_3rd[0]>";
						$to_mail = encode_qp($to_mail);
						$to_mail =~ s/=\r?\n//g;	# Remove the new line added from encode_qp function
	
						#print "Mail Address = $to_mail\n";
						
						(new Mail::Sender)->MailMsg( {smtp => $smtpserver,
									      from => $fromaddress,
									      to => '=?' . $sencoding . '?Q?'.$to_mail.'?=',
									      subject => '=?' . $sencoding . '?Q?'.$subject.'?=',
									      msg => $text_body,
							      	      	      charset => $sencoding} );
					}
				}
				$sth1->finish;
			}; # if (check_mailprefs($row_ary[0], $action)
		}
		$sth->finish;

		if (($action eq 7) or ($action eq 8))
		{
			### Time to send emails to assignment-removal users ###

			my $metr= 0;
			my $email_account;
			while ($metr < scalar(@$users))
			{
				if (check_mailprefs($dbh, $$users[$metr], $action)) {
					# Send mail only if enabled by the user.
					($email_account, $fullname) = bring_me_emailaccount($dbh, $$users[$metr]);
				
					if (!($email_account eq 'unknown@space.gr'))
					{
						if ((!($temp_email_1st[0] eq $email_account)) and (!($temp_email_2nd[0] eq $email_account)) and (!($temp_email_3rd[0] eq $email_account)))
						{
							$to_mail = "$fullname <$email_account>";
							$to_mail = encode_qp($to_mail);
							$to_mail =~ s/=\r?\n//g;	# Remove the new line added from encode_qp function
							(new Mail::Sender)->MailMsg( {smtp => $smtpserver,
										      from => $fromaddress,
										      to => '=?' . $sencoding . '?Q?'.$to_mail.'?=',
										      subject => '=?' . $sencoding . '?Q?'.$subject.'?=',
										      msg => $text_body,
										      charset => $sencoding} );
						};
					};
				};
				$metr = $metr + 1;
			};
		};
	} else {
		### Based on the email table send the mails to defined users ###

		my $metr= 0;
		my $email_account;
		
		while ($metr < scalar(@$users))
		{
			if (check_mailprefs($dbh, $$users[$metr], $action)) {
				# Send mail only if enabled by the user.
				### Send mail to the current mail account ###
			
				($email_account, $fullname) = bring_me_emailaccount($dbh, $$users[$metr]);

				if (!($email_account eq 'unknown@space.gr'))
				{
					$to_mail = "$fullname <$email_account>";
					$to_mail = encode_qp($to_mail);
					$to_mail =~ s/=\r?\n//g;	# Remove the new line added from encode_qp function
					
					(new Mail::Sender)->MailMsg( {smtp => $smtpserver,
								      from => $fromaddress,
								      to => '=?' . $sencoding . '?Q?'.$to_mail.'?=',
								      subject => '=?' . $sencoding . '?Q?'.$subject.'?=',
								      msg => $text_body,
							      	      charset => $sencoding} );
				};
			};
			$metr = $metr + 1;
		};
	};
return;
exit;
};



### Subroutine which takes the user id and returns the email account ###

sub bring_me_emailaccount
{
my ($dbh, $user_id) = @_;

	my $dst = "SELECT u.email, u.fullname
		   FROM st_users u WHERE u.id = $user_id AND u.active = TRUE;";
        
	my $sth = $dbh->prepare($dst);
	$sth->execute;

	if (st_lib::sthErr($sth->err, $sth->errstr)) { 
		# Failed 
		print "Failed !!!";
		$dbh->disconnect;
		exit;
	};

	my @row_ary;
	if (@row_ary = $sth->fetchrow_array) 
	{
		$sth->finish;
		return ($row_ary[0], $row_ary[1]);	
	}
	else
	{
		$sth->finish;
		my $unknown_mail = 'unknown@space.gr';
		my $unknown_fullname = 'unknown_fullname';
		return ($unknown_mail, $unknown_fullname); 
	}
}



### Subroutine - Case of new entry in st_faultsmslog (the text should be included) ###

sub case_sub_st_faultsmsglog
{

my ($dbh, $fault_id) = @_;
my ($temp_id, $temp_description, $temp_log_time, $temp_logged_by_user_id, $temp_username, $temp_fault_shortdescr);

	my $dst = "SELECT u.id, u.description, u.log_time, u.logged_by_user_id, u.fault_id
		   FROM st_faultsmsglog u;";
        
	my $sth = $dbh->prepare($dst);
	$sth->execute;

	if (st_lib::sthErr($sth->err, $sth->errstr)) { 
		# Failed 
		print "Failed !!!";
		$dbh->disconnect;
		exit;
	};

	my @row_ary;
	while (@row_ary = $sth->fetchrow_array) 
	{
		if ($row_ary[4] eq $fault_id)
		{
			$temp_id = $row_ary[0];
			$temp_description = $row_ary[1];
			$temp_log_time = $row_ary[2];
			$temp_logged_by_user_id = $row_ary[3];
		}
	}

	$dst = "SELECT u.fullname
	   	FROM st_users u WHERE u.id = $temp_logged_by_user_id;";
        
	$sth = $dbh->prepare($dst);
	$sth->execute;

	if (st_lib::sthErr($sth->err, $sth->errstr)) { 
		# Failed 
		print "Failed !!!";
		$dbh->disconnect;
		exit;
	};
	
	if (@row_ary = $sth->fetchrow_array) 	{	$temp_username = $row_ary[0];	}
	else					{	$temp_username = "Unknown User"	}

	$dst = "SELECT u.shortdescr
	   	FROM st_faults u WHERE u.id = $fault_id;";
        
	$sth = $dbh->prepare($dst);
	$sth->execute;

	if (st_lib::sthErr($sth->err, $sth->errstr)) { 
		# Failed 
		print "Failed !!!";
		$dbh->disconnect;
		exit;
	};
	
	if (@row_ary = $sth->fetchrow_array) 	{	$temp_fault_shortdescr = $row_ary[0];	 	}
	else					{	$temp_fault_shortdescr = "Unknown SR"	}


	my $subject = "New log entry (f. #$fault_id): $temp_fault_shortdescr";
	my $body_text = "User: " . $temp_username .  "\nLog time: " . st_lib::datetostr($temp_log_time) .	
			"\n\nNew log entry:\n\n" . $temp_description;
	$body_text = $body_text . "\n\nSR description: " . $temp_fault_shortdescr;
	
	return ($subject, $body_text);
}



### Subroutine - Case of new entry in st_faultslevellog ###

sub case_sub_st_faultslevellog
{

my ($dbh, $fault_id) = @_;
my ($temp_id, $temp_change_time, $temp_faultlevel_id, $temp_changed_by_user_id, $temp_username, $temp_level_name, $temp_level_descr, $temp_fault_shortdescr);

	my $dst = "SELECT u.id, u.change_time, u.faultlevel_id, u.changed_by_user_id, u.fault_id
		   FROM st_faultslevellog u;";
        
	my $sth = $dbh->prepare($dst);
	$sth->execute;

	if (st_lib::sthErr($sth->err, $sth->errstr)) { 
		# Failed 
		print "Failed !!!";
		$dbh->disconnect;
		exit;
	};

	my @row_ary;
	while (@row_ary = $sth->fetchrow_array) 
	{
		if ($row_ary[4] eq $fault_id)
		{
			$temp_id = $row_ary[0];
			$temp_change_time = $row_ary[1];
			$temp_faultlevel_id = $row_ary[2];
			$temp_changed_by_user_id = $row_ary[3];
		}
	}

	$dst = "SELECT u.fullname
	   	FROM st_users u WHERE u.id = $temp_changed_by_user_id;";
        
	$sth = $dbh->prepare($dst);
	$sth->execute;

	if (st_lib::sthErr($sth->err, $sth->errstr)) { 
		# Failed 
		print "Failed !!!";
		$dbh->disconnect;
		exit;
	};
	
	if (@row_ary = $sth->fetchrow_array) 	{	$temp_username = $row_ary[0];	}
	else					{	$temp_username = "Unknown User"	}

	$dst = "SELECT u.name, u.description
	   	FROM st_faultlevels u WHERE u.id = $temp_faultlevel_id;";
        
	$sth = $dbh->prepare($dst);
	$sth->execute;

	if (st_lib::sthErr($sth->err, $sth->errstr)) { 
		# Failed 
		print "Failed !!!";
		$dbh->disconnect;
		exit;
	};
	
	if (@row_ary = $sth->fetchrow_array) 	
	{	
		$temp_level_name = $row_ary[0];	
		$temp_level_descr = $row_ary[1];
	}
	else					
	{	
		$temp_level_name = "Unknown level name";
		$temp_level_descr = "Unknown level description";
	}

	$dst = "SELECT u.shortdescr
	   	FROM st_faults u WHERE u.id = $fault_id;";
        
	$sth = $dbh->prepare($dst);
	$sth->execute;

	if (st_lib::sthErr($sth->err, $sth->errstr)) { 
		# Failed 
		print "Failed !!!";
		$dbh->disconnect;
		exit;
	};
	
	if (@row_ary = $sth->fetchrow_array) 	{	$temp_fault_shortdescr = $row_ary[0];	 	}
	else					{	$temp_fault_shortdescr = "Unknown fault"	}

	
	my $subject = "New SR severity (SR #$fault_id): $temp_fault_shortdescr";
	my $body_text = "User: " . $temp_username . "\nChange date: " . st_lib::datetostr($temp_change_time) .
	                "\n\nNew SR severity: " . $temp_level_name;
	$body_text = $body_text . "\n\nSR description: $temp_fault_shortdescr";

	return ($subject, $body_text);
}



### Subroutine - Case of new entry in st_faultsstatuslog ###

sub case_sub_st_faultsstatuslog
{

my ($dbh, $fault_id) = @_;
my ($temp_id, $temp_change_time, $temp_faultstatus_id, $temp_changed_by_user_id, $temp_username, $temp_status_name, $temp_fault_shortdescr);

	my $dst = "SELECT u.id, u.change_time, u.fault_id, u.faultstatus_id, u.changed_by_user_id
		   FROM st_faultsstatuslog u;";
        
	my $sth = $dbh->prepare($dst);
	$sth->execute;

	if (st_lib::sthErr($sth->err, $sth->errstr)) { 
		# Failed 
		print "Failed !!!";
		$dbh->disconnect;
		exit;
	};

	my @row_ary;
	while (@row_ary = $sth->fetchrow_array) 
	{
		if ($row_ary[2] eq $fault_id)
		{
			$temp_id = $row_ary[0];
			$temp_change_time = $row_ary[1];
			$temp_faultstatus_id = $row_ary[3];
			$temp_changed_by_user_id = $row_ary[4];
		}
	}

	$dst = "SELECT u.fullname
	   	FROM st_users u WHERE u.id = $temp_changed_by_user_id;";
        
	$sth = $dbh->prepare($dst);
	$sth->execute;

	if (st_lib::sthErr($sth->err, $sth->errstr)) { 
		# Failed 
		print "Failed !!!";
		$dbh->disconnect;
		exit;
	};
	
	if (@row_ary = $sth->fetchrow_array) 	{	$temp_username = $row_ary[0];	}
	else					{	$temp_username = "Unknown User"	}

	$dst = "SELECT u.name
	   	FROM st_faultstatus u WHERE u.id = $temp_faultstatus_id;";
        
	$sth = $dbh->prepare($dst);
	$sth->execute;

	if (st_lib::sthErr($sth->err, $sth->errstr)) { 
		# Failed 
		print "Failed !!!";
		$dbh->disconnect;
		exit;
	};
	
	if (@row_ary = $sth->fetchrow_array) 	
	{	$temp_status_name = $row_ary[0];		}
	else					
	{	$temp_status_name = "Unknown status name";	}

	$dst = "SELECT u.shortdescr
	   	FROM st_faults u WHERE u.id = $fault_id;";
        
	$sth = $dbh->prepare($dst);
	$sth->execute;

	if (st_lib::sthErr($sth->err, $sth->errstr)) { 
		# Failed 
		print "Failed !!!";
		$dbh->disconnect;
		exit;
	};
	
	if (@row_ary = $sth->fetchrow_array) 	{	$temp_fault_shortdescr = $row_ary[0];	 	}
	else					{	$temp_fault_shortdescr = "Unknown SR"	}

	my $subject = "SR status change (SR #$fault_id): $temp_fault_shortdescr";
	my $body_text = "User: " . $temp_username .  "\nChange date: " . st_lib::datetostr($temp_change_time) .
			"\n\nNew SR status: " . $temp_status_name;
	$body_text = $body_text . "\n\nSR description: $temp_fault_shortdescr";

	return ($subject, $body_text);
}



### Subroutine - Case of new entry in st_faultstechs ###

sub case_sub_st_faultstechs
{

my ($dbh, $fault_id) = @_;
my ($temp_id, $temp_assign_time, $temp_assigned_by_user_id, $temp_assigned_to_user_id, $temp_fault_shortdescr, $temp_by_username, $temp_to_username);

	my $dst = "SELECT u.id, u.assign_time, u.assigned_by_user_id, u.assigned_to_user_id, u.fault_id
		   FROM st_faultstechs u;";
        
	my $sth = $dbh->prepare($dst);
	$sth->execute;

	if (st_lib::sthErr($sth->err, $sth->errstr)) { 
		# Failed 
		print "Failed !!!";
		$dbh->disconnect;
		exit;
	};

	my @row_ary;
	while (@row_ary = $sth->fetchrow_array) 
	{
		if ($row_ary[4] eq $fault_id)
		{
			$temp_id = $row_ary[0];
			$temp_assign_time = $row_ary[1];
			$temp_assigned_by_user_id = $row_ary[2];
			$temp_assigned_to_user_id = $row_ary[3];
		}
	}

	$dst = "SELECT u.fullname
	   	FROM st_users u WHERE u.id = $temp_assigned_by_user_id;";
        
	$sth = $dbh->prepare($dst);
	$sth->execute;

	if (st_lib::sthErr($sth->err, $sth->errstr)) { 
		# Failed 
		print "Failed !!!";
		$dbh->disconnect;
		exit;
	};
	
	if (@row_ary = $sth->fetchrow_array) 	{	$temp_by_username = $row_ary[0];	}
	else					{	$temp_by_username = "Unknown User"	}

	$dst = "SELECT u.fullname
	   	FROM st_users u WHERE u.id = $temp_assigned_to_user_id;";
        
	$sth = $dbh->prepare($dst);
	$sth->execute;

	if (st_lib::sthErr($sth->err, $sth->errstr)) { 
		# Failed 
		print "Failed !!!";
		$dbh->disconnect;
		exit;
	};
	
	if (@row_ary = $sth->fetchrow_array) 	{	$temp_to_username = $row_ary[0];	}
	else					{	$temp_to_username = "Unknown User"	}

	$dst = "SELECT u.shortdescr
	   	FROM st_faults u WHERE u.id = $fault_id;";
        
	$sth = $dbh->prepare($dst);
	$sth->execute;

	if (st_lib::sthErr($sth->err, $sth->errstr)) { 
		# Failed 
		print "Failed !!!";
		$dbh->disconnect;
		exit;
	};
	
	if (@row_ary = $sth->fetchrow_array) 	{	$temp_fault_shortdescr = $row_ary[0];	 	}
	else					{	$temp_fault_shortdescr = "Unknown SR"	}

	my $subject = "SR assignment (SR #$fault_id): $temp_fault_shortdescr";
	my $body_text = "Assignment time: " . st_lib::datetostr($temp_assign_time) . "\nAssigned by <" . $temp_by_username . "> to <" . $temp_to_username . ">";
	$body_text = $body_text . "\n\nSR description: $temp_fault_shortdescr";
	
	return ($subject, $body_text);
}



### Subroutine - Case of close a fault ###

sub case_sub_st_faultclosed
{

my ($dbh, $fault_id) = @_;
my ($temp_id, $temp_shortdescr, $temp_start_time, $temp_end_time, $temp_logged_by_user_id, $temp_closed_by_user_id, $temp_companylocation_id, $temp_faulttype_id, $temp_logged_username, $temp_closed_username, $temp_companylocationname, $temp_companylocationdescr, $temp_faulttype_name, $temp_faulttype_descr, $temp_company_id, $temp_company_name);

	my $dst = "SELECT u.id, u.shortdescr, u.start_time, u.end_time, u.logged_by_user_id, u.closed_by_user_id, u.companylocation_id, u.faulttype_id
		   FROM st_faults u WHERE u.id = $fault_id;";
        
	my $sth = $dbh->prepare($dst);
	$sth->execute;

	if (st_lib::sthErr($sth->err, $sth->errstr)) { 
		# Failed 
		print "Failed !!!";
		$dbh->disconnect;
		exit;
	};

	my @row_ary;
	if (@row_ary = $sth->fetchrow_array) 
	{
		$temp_id = $row_ary[0];
		$temp_shortdescr = $row_ary[1];
		$temp_start_time = $row_ary[2];
		$temp_end_time =  $row_ary[3];
		$temp_logged_by_user_id = $row_ary[4];
		$temp_closed_by_user_id = $row_ary[5];
		$temp_companylocation_id = $row_ary[6];
		$temp_faulttype_id = $row_ary[7];
	}

	$dst = "SELECT u.fullname
	   	FROM st_users u WHERE u.id = $temp_logged_by_user_id;";
        
	$sth = $dbh->prepare($dst);
	$sth->execute;

	if (st_lib::sthErr($sth->err, $sth->errstr)) { 
		# Failed 
		print "Failed !!!";
		$dbh->disconnect;
		exit;
	};
	
	if (@row_ary = $sth->fetchrow_array) 	{	$temp_logged_username = $row_ary[0];	}
	else					{	$temp_logged_username = "Unknown User"	}

	$dst = "SELECT u.fullname
	   	FROM st_users u WHERE u.id = $temp_closed_by_user_id;";
        
	$sth = $dbh->prepare($dst);
	$sth->execute;

	if (st_lib::sthErr($sth->err, $sth->errstr)) { 
		# Failed 
		print "Failed !!!";
		$dbh->disconnect;
		exit;
	};
	
	if (@row_ary = $sth->fetchrow_array) 	{	$temp_closed_username = $row_ary[0];	}
	else					{	$temp_closed_username = "Unknown User"	}
	
	$dst = "SELECT u.name, u.description, u.company_id
	   	FROM st_companieslocations u WHERE u.id = $temp_companylocation_id;";
        
	$sth = $dbh->prepare($dst);
	$sth->execute;

	if (st_lib::sthErr($sth->err, $sth->errstr)) { 
		# Failed 
		print "Failed !!!";
		$dbh->disconnect;
		exit;
	};
	
	if (@row_ary = $sth->fetchrow_array) 	
	{	
		$temp_companylocationname = $row_ary[0];
		$temp_companylocationdescr = $row_ary[1];
		$temp_company_id = $row_ary[2];
	}
	else
	{	
		$temp_companylocationname = "Unknown company location name";
		$temp_companylocationdescr = "Unknown company location description";
		$temp_company_id = "Unknown company id";
	}

	$dst = "SELECT u.name
	   	FROM st_companies u WHERE u.id = $temp_company_id;";
        
	$sth = $dbh->prepare($dst);
	$sth->execute;

	if (st_lib::sthErr($sth->err, $sth->errstr)) { 
		# Failed 
		print "Failed !!!";
		$dbh->disconnect;
		exit;
	};
	
	if (@row_ary = $sth->fetchrow_array) 	{	$temp_company_name = $row_ary[0];		}
	else					{	$temp_company_name = "Unknown company name"	}

	$dst = "SELECT u.name, u.description
	   	FROM st_faulttypes u WHERE u.id = $temp_faulttype_id;";
        
	$sth = $dbh->prepare($dst);
	$sth->execute;

	if (st_lib::sthErr($sth->err, $sth->errstr)) { 
		# Failed 
		print "Failed !!!";
		$dbh->disconnect;
		exit;
	};
	
	if (@row_ary = $sth->fetchrow_array) 	
	{	
		$temp_faulttype_name = $row_ary[0];	
		$temp_faulttype_descr = $row_ary[1];
	}
	else					
	{	
		$temp_faulttype_name = "Unknown SR type name";
		$temp_faulttype_descr = "Unknown SR type description";
	}

	my $subject = "SR closed (SR #$fault_id): $temp_shortdescr";
	my $body_text = "SR type: " . $temp_faulttype_name . "\nStart time: " . st_lib::datetostr($temp_start_time) . "  Closing time: " . st_lib::datetostr($temp_end_time) . "\n\nSR owned by: " . $temp_logged_username . "\nSR closed by: " . $temp_closed_username . "\n\nCompany: $temp_company_name" . "  Company location: " . $temp_companylocationname;

	### Check if the specific ticket type *has* a survey and 'add' the proper link ###
	
	$dst = "SELECT u.surv_id, u.type_id
	   	FROM st_surv2types u WHERE u.type_id = $temp_faulttype_id;";
        
	$sth = $dbh->prepare($dst);
	$sth->execute;

	if (st_lib::sthErr($sth->err, $sth->errstr)) { 
		# Failed 
		print "Failed !!!";
		$dbh->disconnect;
		exit;
	};
	
	if (@row_ary = $sth->fetchrow_array) 	
	{
		$body_text = $body_text . "\n\n" . "Please fill out our satisfaction survey about this case: $url+$fault_id";
	}
	
	return ($subject, $body_text);
}



### Subroutine - Case of open a fault ###

sub case_sub_st_faultopened
{

my ($dbh, $fault_id) = @_;
my ($temp_id, $temp_shortdescr, $temp_start_time, $temp_end_time, $temp_logged_by_user_id, $temp_companylocation_id, $temp_faulttype_id, $temp_logged_username, $temp_companylocationname, $temp_companylocationdescr, $temp_faulttype_name, $temp_faulttype_descr, $temp_company_id, $temp_company_name, $temp_descr_firstlogentry);

	my $dst = "SELECT u.id, u.shortdescr, u.start_time, u.end_time, u.logged_by_user_id, u.companylocation_id, u.faulttype_id
		   FROM st_faults u WHERE u.id = $fault_id;";
        
	my $sth = $dbh->prepare($dst);
	$sth->execute;

	if (st_lib::sthErr($sth->err, $sth->errstr)) { 
		# Failed 
		print "Failed !!!";
		$dbh->disconnect;
		exit;
	};

	my @row_ary;
	if (@row_ary = $sth->fetchrow_array) 
	{
		$temp_id = $row_ary[0];
		$temp_shortdescr = $row_ary[1];
		$temp_start_time = $row_ary[2];
		$temp_end_time =  $row_ary[3];
		$temp_logged_by_user_id = $row_ary[4];
		$temp_companylocation_id = $row_ary[5];
		$temp_faulttype_id = $row_ary[6];
	}

	$dst = "SELECT u.fullname
	   	FROM st_users u WHERE u.id = $temp_logged_by_user_id;";
        
	$sth = $dbh->prepare($dst);
	$sth->execute;

	if (st_lib::sthErr($sth->err, $sth->errstr)) { 
		# Failed 
		print "Failed !!!";
		$dbh->disconnect;
		exit;
	};
	
	if (@row_ary = $sth->fetchrow_array) 	{	$temp_logged_username = $row_ary[0];	}
	else					{	$temp_logged_username = "Unknown User"	}

	$dst = "SELECT u.name, u.description, u.company_id
	   	FROM st_companieslocations u WHERE u.id = $temp_companylocation_id;";
        
	$sth = $dbh->prepare($dst);
	$sth->execute;

	if (st_lib::sthErr($sth->err, $sth->errstr)) { 
		# Failed 
		print "Failed !!!";
		$dbh->disconnect;
		exit;
	};
	
	if (@row_ary = $sth->fetchrow_array) 	
	{	
		$temp_companylocationname = $row_ary[0];
		$temp_companylocationdescr = $row_ary[1];
		$temp_company_id = $row_ary[2];
	}
	else
	{	
		$temp_companylocationname = "Unknown company location name";
		$temp_companylocationdescr = "Unknown company location description";
		$temp_company_id = "Unknown company id";
	}

	$dst = "SELECT u.name
	   	FROM st_companies u WHERE u.id = $temp_company_id;";
        
	$sth = $dbh->prepare($dst);
	$sth->execute;

	if (st_lib::sthErr($sth->err, $sth->errstr)) { 
		# Failed 
		print "Failed !!!";
		$dbh->disconnect;
		exit;
	};
	
	if (@row_ary = $sth->fetchrow_array) 	{	$temp_company_name = $row_ary[0];		}
	else					{	$temp_company_name = "Unknown company name"	}

	$dst = "SELECT u.name, u.description
	   	FROM st_faulttypes u WHERE u.id = $temp_faulttype_id;";
        
	$sth = $dbh->prepare($dst);
	$sth->execute;

	if (st_lib::sthErr($sth->err, $sth->errstr)) { 
		# Failed 
		print "Failed !!!";
		$dbh->disconnect;
		exit;
	};
	
	if (@row_ary = $sth->fetchrow_array) 	
	{	
		$temp_faulttype_name = $row_ary[0];	
		$temp_faulttype_descr = $row_ary[1];
	}
	else					
	{	
		$temp_faulttype_name = "Unknown SR type name";
		$temp_faulttype_descr = "Unknown SR type description";
	}

	$dst = "SELECT u.description FROM st_faultsmsglog u WHERE u.fault_id = $fault_id;";
        
	$sth = $dbh->prepare($dst);
	$sth->execute;

	if (st_lib::sthErr($sth->err, $sth->errstr)) { 
		# Failed 
		print "Failed !!!";
		$dbh->disconnect;
		exit;
	};

	if (@row_ary = $sth->fetchrow_array) 
	{
		$temp_descr_firstlogentry = $row_ary[0];
	}

	
	my $subject = "New Service Request (#$fault_id): $temp_shortdescr";
	my $body_text = "SR type: " . $temp_faulttype_name . "\nStart time: " . st_lib::datetostr($temp_start_time) . "\n\nCreated by: " . $temp_logged_username . "\n\nCompany: $temp_company_name" . "  Company location: " . $temp_companylocationname;

	$body_text = $body_text . "\n\nSR description: " . $temp_descr_firstlogentry;
	
	return ($subject, $body_text);
}



### Subroutine - Case of related users assignment ###

sub case_sub_st_fault_rel_assignment
{

my ($dbh, $fault_id, $userid) = @_;
my ($temp_id, $temp_shortdescr, $temp_start_time, $temp_end_time, $temp_logged_by_user_id, $temp_companylocation_id, $temp_faulttype_id, $temp_logged_username, $temp_companylocationname, $temp_companylocationdescr, $temp_faulttype_name, $temp_faulttype_descr, $temp_company_id, $temp_company_name, $temp_fullname, $temp_username);

	my $dst = "SELECT u.id, u.shortdescr, u.start_time, u.end_time, u.logged_by_user_id, u.companylocation_id, u.faulttype_id
		   FROM st_faults u WHERE u.id = $fault_id;";
        
	my $sth = $dbh->prepare($dst);
	$sth->execute;

	if (st_lib::sthErr($sth->err, $sth->errstr)) { 
		# Failed 
		print "Failed !!!";
		$dbh->disconnect;
		exit;
	};

	my @row_ary;
	if (@row_ary = $sth->fetchrow_array) 
	{
		$temp_id = $row_ary[0];
		$temp_shortdescr = $row_ary[1];
		$temp_start_time = $row_ary[2];
		$temp_end_time =  $row_ary[3];
		$temp_logged_by_user_id = $row_ary[4];
		$temp_companylocation_id = $row_ary[5];
		$temp_faulttype_id = $row_ary[6];
	}

	$dst = "SELECT u.fullname, u.username
	   	FROM st_users u WHERE u.id = $userid;";
        
	$sth = $dbh->prepare($dst);
	$sth->execute;

	if (st_lib::sthErr($sth->err, $sth->errstr)) { 
		# Failed 
		print "Failed !!!";
		$dbh->disconnect;
		exit;
	};
	
	if (@row_ary = $sth->fetchrow_array) 	{	$temp_fullname = $row_ary[0]; $temp_username = $row_ary[1];	}
	else					{	$temp_fullname = "Unknown User"; $temp_username = "Unknown User";	}

	$dst = "SELECT u.fullname
	   	FROM st_users u WHERE u.id = $temp_logged_by_user_id;";
        
	$sth = $dbh->prepare($dst);
	$sth->execute;

	if (st_lib::sthErr($sth->err, $sth->errstr)) { 
		# Failed 
		print "Failed !!!";
		$dbh->disconnect;
		exit;
	};
	
	if (@row_ary = $sth->fetchrow_array) 	{	$temp_logged_username = $row_ary[0];	}
	else					{	$temp_logged_username = "Unknown User"	}

	$dst = "SELECT u.name, u.description, u.company_id
	   	FROM st_companieslocations u WHERE u.id = $temp_companylocation_id;";
        
	$sth = $dbh->prepare($dst);
	$sth->execute;

	if (st_lib::sthErr($sth->err, $sth->errstr)) { 
		# Failed 
		print "Failed !!!";
		$dbh->disconnect;
		exit;
	};
	
	if (@row_ary = $sth->fetchrow_array) 	
	{	
		$temp_companylocationname = $row_ary[0];
		$temp_companylocationdescr = $row_ary[1];
		$temp_company_id = $row_ary[2];
	}
	else
	{	
		$temp_companylocationname = "Unknown company location name";
		$temp_companylocationdescr = "Unknown company location description";
		$temp_company_id = "Unknown company id";
	}

	$dst = "SELECT u.name
	   	FROM st_companies u WHERE u.id = $temp_company_id;";
        
	$sth = $dbh->prepare($dst);
	$sth->execute;

	if (st_lib::sthErr($sth->err, $sth->errstr)) { 
		# Failed 
		print "Failed !!!";
		$dbh->disconnect;
		exit;
	};
	
	if (@row_ary = $sth->fetchrow_array) 	{	$temp_company_name = $row_ary[0];		}
	else					{	$temp_company_name = "Unknown company name"	}

	$dst = "SELECT u.name, u.description
	   	FROM st_faulttypes u WHERE u.id = $temp_faulttype_id;";
        
	$sth = $dbh->prepare($dst);
	$sth->execute;

	if (st_lib::sthErr($sth->err, $sth->errstr)) { 
		# Failed 
		print "Failed !!!";
		$dbh->disconnect;
		exit;
	};
	
	if (@row_ary = $sth->fetchrow_array) 	
	{	
		$temp_faulttype_name = $row_ary[0];	
		$temp_faulttype_descr = $row_ary[1];
	}
	else					
	{	
		$temp_faulttype_name = "Unknown SR type name";
		$temp_faulttype_descr = "Unknown SR type description";
	}

	# Check credentials, if not ok call GetUserForLogin
	my $query = new CGI;
	my ($zuser, $zpass, $zfullname, @zdata) = st_lib::GetCredentials($query);

	my $subject = "Related user added (SR #$fault_id): $temp_shortdescr";
	my $body_text = "User added from (fullname): $zfullname (username: $zuser)\n" . "Added user: $temp_fullname (username: $temp_username)\nSR type: " . $temp_faulttype_name . "\nStart time: " . st_lib::datetostr($temp_start_time) . "\n\nSR owned by: " . $temp_logged_username . "\n\nCompany: $temp_company_name" . "  Company location: " . $temp_companylocationname;
	
	return ($subject, $body_text);
}



### Subroutine - Case of related users removal ###

sub case_sub_st_fault_rel_removal
{

my ($dbh, $fault_id, $users) = @_;
my ($temp_id, $temp_shortdescr, $temp_start_time, $temp_end_time, $temp_logged_by_user_id, $temp_companylocation_id, $temp_faulttype_id, $temp_logged_username, $temp_companylocationname, $temp_companylocationdescr, $temp_faulttype_name, $temp_faulttype_descr, $temp_company_id, $temp_company_name, $temp_fullname, $metr);

	my $dst = "SELECT u.id, u.shortdescr, u.start_time, u.end_time, u.logged_by_user_id, u.companylocation_id, u.faulttype_id
		   FROM st_faults u WHERE u.id = $fault_id;";
        
	my $sth = $dbh->prepare($dst);
	$sth->execute;

	if (st_lib::sthErr($sth->err, $sth->errstr)) { 
		# Failed 
		print "Failed !!!";
		$dbh->disconnect;
		exit;
	};

	my @row_ary;
	if (@row_ary = $sth->fetchrow_array) 
	{
		$temp_id = $row_ary[0];
		$temp_shortdescr = $row_ary[1];
		$temp_start_time = $row_ary[2];
		$temp_end_time =  $row_ary[3];
		$temp_logged_by_user_id = $row_ary[4];
		$temp_companylocation_id = $row_ary[5];
		$temp_faulttype_id = $row_ary[6];
	}

	$dst = "SELECT u.fullname
	   	FROM st_users u WHERE u.id = $temp_logged_by_user_id;";
        
	$sth = $dbh->prepare($dst);
	$sth->execute;

	if (st_lib::sthErr($sth->err, $sth->errstr)) { 
		# Failed 
		print "Failed !!!";
		$dbh->disconnect;
		exit;
	};
	
	if (@row_ary = $sth->fetchrow_array) 	{	$temp_logged_username = $row_ary[0];	}
	else					{	$temp_logged_username = "Unknown User"	}

	$dst = "SELECT u.name, u.description, u.company_id
	   	FROM st_companieslocations u WHERE u.id = $temp_companylocation_id;";
        
	$sth = $dbh->prepare($dst);
	$sth->execute;

	if (st_lib::sthErr($sth->err, $sth->errstr)) { 
		# Failed 
		print "Failed !!!";
		$dbh->disconnect;
		exit;
	};
	
	if (@row_ary = $sth->fetchrow_array) 	
	{	
		$temp_companylocationname = $row_ary[0];
		$temp_companylocationdescr = $row_ary[1];
		$temp_company_id = $row_ary[2];
	}
	else
	{	
		$temp_companylocationname = "Unknown company location name";
		$temp_companylocationdescr = "Unknown company location description";
		$temp_company_id = "Unknown company id";
	}

	$dst = "SELECT u.name
	   	FROM st_companies u WHERE u.id = $temp_company_id;";
        
	$sth = $dbh->prepare($dst);
	$sth->execute;

	if (st_lib::sthErr($sth->err, $sth->errstr)) { 
		# Failed 
		print "Failed !!!";
		$dbh->disconnect;
		exit;
	};
	
	if (@row_ary = $sth->fetchrow_array) 	{	$temp_company_name = $row_ary[0];		}
	else					{	$temp_company_name = "Unknown company name"	}

	$dst = "SELECT u.name, u.description
	   	FROM st_faulttypes u WHERE u.id = $temp_faulttype_id;";
        
	$sth = $dbh->prepare($dst);
	$sth->execute;

	if (st_lib::sthErr($sth->err, $sth->errstr)) { 
		# Failed 
		print "Failed !!!";
		$dbh->disconnect;
		exit;
	};
	
	if (@row_ary = $sth->fetchrow_array) 	
	{	
		$temp_faulttype_name = $row_ary[0];	
		$temp_faulttype_descr = $row_ary[1];
	}
	else					
	{	
		$temp_faulttype_name = "Unknown SR type name";
		$temp_faulttype_descr = "Unknown SR type description";
	}
	
	# Check credentials, if not ok call GetUserForLogin
	my $query = new CGI;
	my ($zuser, $zpass, $zfullname, @zdata) = st_lib::GetCredentials($query);
	
	my $subject = "Related user removed (SR #$fault_id): $temp_shortdescr";
	my $body_text = "SR type: " . $temp_faulttype_name . "\nStart time: " . st_lib::datetostr($temp_start_time) . "\n\nSR owned by: " . $temp_logged_username . "\n\nCompany: $temp_company_name" . "  Company location: " . $temp_companylocationname;
	$body_text = $body_text . "\n\nUsers removal from (fullname): $zfullname (username: $zuser)\n" . "The following users are not anymore related:\n";
	$body_text = $body_text . "\n";
	$body_text = $body_text . "\n";
	
	$metr = 0;	
	while ($metr < scalar(@$users))
	{
		$dst = "SELECT u.fullname
		   	FROM st_users u WHERE u.id = $$users[$metr];";
        
		$sth = $dbh->prepare($dst);
		$sth->execute;

		if (st_lib::sthErr($sth->err, $sth->errstr)) { 
			# Failed 
			print "Failed !!!";
			$dbh->disconnect;
			exit;
		};
	
		if (@row_ary = $sth->fetchrow_array) 	{	$temp_fullname = $row_ary[0];	}
		else					{	$temp_fullname = "Unknown User"	}

		$body_text = $body_text . "$temp_fullname\n";
		$metr = $metr + 1;
	}

	return ($subject, $body_text);
}


### Subroutine - Case of answered survey ###

sub case_sub_st_surv_answered {
	# Build mail for answered survey.
	my ($dbh, $fault_id) = @_;
	my $dst = "SELECT a.answer, q.qu_text, q.type, q.value1, q.value2
		FROM st_surv_design q INNER JOIN st_surv_answ a ON q.id = a.qu_id
		WHERE a.fault_id = $fault_id";

	my $sth = $dbh->prepare($dst);
	$sth->execute;
	if (st_lib::sthErr($sth->err, $sth->errstr)) { 
		# Internal Error.
		st_lib::wDbg("ERROR: survey select failed. ", "st_mail::case_sub_st_surv_answered");
		$dbh->disconnect;
		exit;
	};
	my $subject = "Survey answered: SR #$fault_id";
	my $body_text;
	my @row_ary;
	while (@row_ary = $sth->fetchrow_array) {
		$body_text = $body_text . $row_ary[1] . ": ";
		if ($row_ary[2] eq 'num') {
			# Numeric - Radio buttons.
			$body_text = $body_text . $row_ary[0] . " (min=$row_ary[3], max=$row_ary[4])\n";
		} elsif ($row_ary[2] eq 'txt') {
			# Text - Text area
			$body_text = $body_text . $row_ary[0] . "\n";
		};
	};
	return ($subject, $body_text);
};


### Subroutine - Case of file upload ###

sub case_sub_st_file_upload
{

my ($dbh, $file_id) = @_;
my ($temp_filename, $temp_size, $temp_comments, $temp_fault_id);
my ($temp_id, $temp_shortdescr, $temp_start_time, $temp_end_time, $temp_logged_by_user_id, $temp_companylocation_id, $temp_faulttype_id, $temp_logged_username, $temp_companylocationname, $temp_companylocationdescr, $temp_company_id, $temp_company_name, $temp_faulttype_name, $temp_faulttype_descr);

	my $dst = "SELECT u.filename, u.size, u.comments, u.fault_id
		   FROM st_files u WHERE u.id = $file_id;";
        
	my $sth = $dbh->prepare($dst);
	$sth->execute;

	if (st_lib::sthErr($sth->err, $sth->errstr)) { 
		# Failed 
		print "Failed !!!";
		$dbh->disconnect;
		exit;
	};

	my @row_ary;
	if (@row_ary = $sth->fetchrow_array) 
	{
		$temp_filename = $row_ary[0];
		$temp_size = $row_ary[1];
		$temp_comments = $row_ary[2];
		$temp_fault_id =  $row_ary[3];
	}

	$dst = "SELECT u.id, u.shortdescr, u.start_time, u.end_time, u.logged_by_user_id, u.companylocation_id, u.faulttype_id
		FROM st_faults u WHERE u.id = $temp_fault_id;";
        
	$sth = $dbh->prepare($dst);
	$sth->execute;

	if (st_lib::sthErr($sth->err, $sth->errstr)) { 
		# Failed 
		print "Failed !!!";
		$dbh->disconnect;
		exit;
	};

	if (@row_ary = $sth->fetchrow_array) 
	{
		$temp_id = $row_ary[0];
		$temp_shortdescr = $row_ary[1];
		$temp_start_time = $row_ary[2];
		$temp_end_time =  $row_ary[3];
		$temp_logged_by_user_id = $row_ary[4];
		$temp_companylocation_id = $row_ary[5];
		$temp_faulttype_id = $row_ary[6];
	}

	$dst = "SELECT u.fullname
	   	FROM st_users u WHERE u.id = $temp_logged_by_user_id;";
        
	$sth = $dbh->prepare($dst);
	$sth->execute;

	if (st_lib::sthErr($sth->err, $sth->errstr)) { 
		# Failed 
		print "Failed !!!";
		$dbh->disconnect;
		exit;
	};
	
	if (@row_ary = $sth->fetchrow_array) 	{	$temp_logged_username = $row_ary[0];	}
	else					{	$temp_logged_username = "Unknown User"	}

	$dst = "SELECT u.name, u.description, u.company_id
	   	FROM st_companieslocations u WHERE u.id = $temp_companylocation_id;";
        
	$sth = $dbh->prepare($dst);
	$sth->execute;

	if (st_lib::sthErr($sth->err, $sth->errstr)) { 
		# Failed 
		print "Failed !!!";
		$dbh->disconnect;
		exit;
	};
	
	if (@row_ary = $sth->fetchrow_array) 	
	{	
		$temp_companylocationname = $row_ary[0];
		$temp_companylocationdescr = $row_ary[1];
		$temp_company_id = $row_ary[2];
	}
	else
	{	
		$temp_companylocationname = "Unknown company location name";
		$temp_companylocationdescr = "Unknown company location description";
		$temp_company_id = "Unknown company id";
	}

	$dst = "SELECT u.name
	   	FROM st_companies u WHERE u.id = $temp_company_id;";
        
	$sth = $dbh->prepare($dst);
	$sth->execute;

	if (st_lib::sthErr($sth->err, $sth->errstr)) { 
		# Failed 
		print "Failed !!!";
		$dbh->disconnect;
		exit;
	};
	
	if (@row_ary = $sth->fetchrow_array) 	{	$temp_company_name = $row_ary[0];		}
	else					{	$temp_company_name = "Unknown company name"	}

	$dst = "SELECT u.name, u.description
	   	FROM st_faulttypes u WHERE u.id = $temp_faulttype_id;";
        
	$sth = $dbh->prepare($dst);
	$sth->execute;

	if (st_lib::sthErr($sth->err, $sth->errstr)) { 
		# Failed 
		print "Failed !!!";
		$dbh->disconnect;
		exit;
	};
	
	if (@row_ary = $sth->fetchrow_array) 	
	{	
		$temp_faulttype_name = $row_ary[0];	
		$temp_faulttype_descr = $row_ary[1];
	}
	else					
	{	
		$temp_faulttype_name = "Unknown SR type name";
		$temp_faulttype_descr = "Unknown SR type description";
	}

	my $subject = "New File Upload (SR #$temp_fault_id): $temp_shortdescr";
	my $body_text = "The following file has been uploaded:\n\n" . "Filename: " . $temp_filename . "\nFile size: " . $temp_size . " bytes\n\nComments: " . $temp_comments;

	$body_text = $body_text . "\n\nSR type: " . $temp_faulttype_name . "\nSR owned by: " . $temp_logged_username . "\nSR description: " . $temp_shortdescr . "\nCompany: $temp_company_name" . "  Company location: " . $temp_companylocationname;

	return ($subject, $body_text);
}

# check_mailprefs: Returns 0 or 1, given a specific user_id and mail type,
# depending on user's mail preferences.
# Args: user_id, mail type (action) 

sub check_mailprefs {
	my ($dbh, $user_id, $mailtype, $ignore) = @_;
	my %fields = (
		1	=>	'newlogentry',
		2	=>	'severchange',
		3	=>	'statuschang',
		4	=>	'techassignm',
		5	=>	'closedfault',
		6	=>	'addnewfault',
		7	=>	'relusersadd',
		8	=>	'relusersrem',
		9	=>	'filesupload'
	);

	if (not(defined($fields{ $mailtype }))) {
		# No field, return true for safety.
		st_lib::wDbg("DEBUG: Returning 1 for wrong action", "st_mail::check_mailprefs");
		return 1;
	};

	my $ldst = "SELECT " . $fields{ $mailtype } . " FROM st_mailprefs WHERE user_id = $user_id";
	my $lsth = $dbh->prepare($ldst);
	$lsth->execute;

	if (st_lib::sthErr($lsth->err, $lsth->errstr)) { 
		# Failed 
		print "Failed !!!";
		st_lib::wDbg("ERROR: st_mailprefs SELECT Failed ", "st_mail");
		# Return 1 for safety.
		st_lib::wDbg("DEBUG: Returning 1 for failed select", "st_mail::check_mailprefs");
		return 1;
	};
	if ($lsth->rows == 0) {
		# No preferences, so return 1
		st_lib::wDbg("DEBUG: Returning 1 for missing record", "st_mail::check_mailprefs");
		return 1;
	};
	my @lrow = $lsth->fetchrow_array;
	st_lib::wDbg("DEBUG: Returning $lrow[0] for normal", "st_mail::check_mailprefs");
	return $lrow[0]; # This is boolean.
	# We should'nt be here, return 1 for safety.
	st_lib::wDbg("DEBUG: Returning 1 for failover", "st_mail::check_mailprefs");
	return 1;
};

1;
