#!/usr/bin/perl

#############################################################################
# MUDA: Resurrection							    #
# mudaclient.pl		: Client executable.				    #
# Artist:		: Theodore J. Soldatos				    #
#############################################################################
# This file is part of MUDA:Resurrection.			   	    #
#									    #
# MUDA:Resurrection is free software: you can redistribute it and/or modify #
# it under the terms of the GNU General Public License as published by	    #
# the Free Software Foundation, either version 3 of the License, or	    #
# (at your option) any later version.					    #
#									    #
# MUDA:Resurrection is distributed in the hope that it will be useful,	    #
# but WITHOUT ANY WARRANTY; without even the implied warranty of	    #
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the		    #
# GNU General Public License for more details.				    #
#									    #
# You should have received a copy of the GNU General Public License	    #
# along with MUDA:Resurrection.  If not, see <http://www.gnu.org/licenses/>.#
#									    #
# Copyright 2012 Theodore J. Soldatos					    #
#############################################################################


use IO::Async::Stream;
use IO::Async::Timer::Periodic;

use IO::Async::Loop;
use IO::Async::Signal;
use IPC::SysV qw(MSG_NOERROR IPC_RMID IPC_PRIVATE IPC_CREAT S_IRWXU IPC_NOWAIT);
use IO::Select;
use Config::General;


#use Term::ReadLine;
use strict;
no strict "refs";
use warnings;

my $clientPID = $$;
my $serverPID = 0;
my $killrsp;
my $lockdir = '/run/lock/mrc/';
my $maxclients = 3;

# Load configuration:
my $conf = new Config::General("/etc/muda.conf");
my %config = $conf->getall;

if (%config) {
	if (defined $config{lockdir}) {
		$lockdir = $config{lockdir};
	};
	if (defined $config{maxclients}) {
		$maxclients = $config{maxclients};
	};
};

my $mkdrc = mkdir $lockdir;

# Count online clients. Exit early if limit reached:
my $cccnt = 0;
opendir(my $dirh, $lockdir);
while(readdir $dirh) {
	$cccnt++;
};
if ($cccnt >= $maxclients) {
	print "Sorry, maximum clients limit reached. Please retry later.\n";
	# Nothing to cleanup yet, just exit.
	exit;
};

# Identify self:
my $pidfile = "$lockdir$clientPID";
open PIDFILE, ">$pidfile" or die "Could not open pid file!\n";

select(PIDFILE); 
$| = 1;

# link to main message queue:
my $mmqid = msgget(1000000, S_IRWXU);

if (not defined $mmqid) {
	print "Unable to connect to server.\n";
	close PIDFILE;
	unlink $pidfile;
	exit 1;
};

my $loop = IO::Async::Loop->new;

$loop->add( IO::Async::Timer::Periodic->new(
   # This will call the main operational code of client:
   interval => 1,
   on_tick => \&mainMudaTick 
)->start );

$loop->add( IO::Async::Timer::Periodic->new(
   # This will call the main operational code of client:
   interval => 1,
   first_interval => 10,
   on_tick => \&checkAlive 
)->start );

$loop->add( IO::Async::Stream->new(
	read_handle => \*STDIN, 
	write_handle => \*STDOUT,
    	on_read => \&consoleInput, 
 	) 
);

my $signal = IO::Async::Signal->new(
    name => "USR1",
    on_receipt => \&mainMudaTick
 );

 $loop->add( $signal );


# Initialize:
my $pos = 1;	# "User" position.
my $timecntr = 0;
my $mqmsgtype = 1;

my $inact_timer = time;	# Inactivity timer.
my $inact_warn = 290;	# Warn after these seconds.
my $inact_logoff = 300;	# Logoff after these seconds.
select(STDOUT); # default
$| = 1;

# Login: 

my $prompt = ">";

my $cmdArgs;

my $maxtries = 3;
my $loggedin = 0;
# Create my queue:
my $cmqid = msgget($clientPID, IPC_CREAT | S_IRWXU);

$SIG{INT} = sub {
	print "\nDisconnecting...\n";
	msgctl($cmqid, IPC_RMID, 0);
	close PIDFILE;
	unlink $pidfile;
	exit;
	};

print " _ _________________________________________________\n";
print "/ \\                                                 \\\n";
print "|  |                                                 \\\n";
print "\\__|                                                  |\n";
print "   >--------------------------------------------------<\n";
print "   |                                                  |\n";
print "   |    MUDA:Resurrection - The return of the Code    |\n";
print "   |                                                  |\n";
print "   |    /-------------------------------------------------/\n";
print "   |   /                                                 /\n";
print "    \\_/_________________________________________________/\n";
print "\n";
print "\n";

while (($maxtries > 0) and ($loggedin == 0)) {
	my ($username, $password) = '';
	print "Username: ";
	eval {
		local $SIG{ALRM} = sub { die "alarm\n" }; # NB: \n required
		alarm 10;
		$username = <STDIN>;
		alarm 0;
    	};
	if ($@) {
		die unless $@ eq "alarm\n"; # propagate unexpected errors
		# timed out
		print "Timeout...\n";
		msgctl($cmqid, IPC_RMID, 0);
		close PIDFILE;
		unlink $pidfile;
		exit 0;
	}; 
	$username = '' if not $username;
	$username =~ s/[\r\n]+$//;
	$username = quotemeta($username);
	if ($username eq '') {
		print "Goodbye...\n";
		msgctl($cmqid, IPC_RMID, 0);
		close PIDFILE;
		unlink $pidfile;
		exit 0;
	};
	print "Password: ";
	eval {
		local $SIG{ALRM} = sub { die "alarm\n" }; # NB: \n required
		alarm 30;
		$password = <STDIN>;
		alarm 0;
    	};
	if ($@) {
		die unless $@ eq "alarm\n"; # propagate unexpected errors
		# timed out
		print "Timeout...\n";
		msgctl($cmqid, IPC_RMID, 0);
		close PIDFILE;
		unlink $pidfile;
		exit 0;
	}; 

	$password = '' if not $password;
	$password =~ s/[\r\n]+$//;
	#$password = quotemeta($password);
	if ($password eq '') {
		print "Empty password not allowed. Goodbye...\n";
		msgctl($cmqid, IPC_RMID, 0);
		close PIDFILE;
		unlink $pidfile;
		exit 0;
	};
	chomp $username;
	chomp $password;

	# Inform server through main queue:
	my $message = "$clientPID cmdLogin $username $password";
	my $bytes = do {use bytes; length($message)};
	if ($bytes > 2000) {
		$message = substr $message, 0, 2000;
	};
	my $mqmsg = pack("l! a*", $mqmsgtype, $message);
	msgsnd($mmqid, $mqmsg, IPC_NOWAIT);
	my $mqrcv;
	print "\nLogging in...";
	my $rcvrc = msgrcv($cmqid, $mqrcv, 2048, 0, MSG_NOERROR);
	if (not defined $rcvrc) {
		print "Error receiving from server (strange).\n";
		msgctl($cmqid, IPC_RMID, 0);
		close PIDFILE;
		unlink $pidfile;
		exit 0;
	};
	my ($typ_rcvd, $rcvdData) = unpack("l! a*", $mqrcv);
	if ($rcvdData eq '0') {
		print "Login failed. " . --$maxtries . " remaining.\n"; 
	} elsif ($rcvdData eq '-1') {
		$maxtries = 0;
		print "Login failed. Already logged in.\n";
	} else {
		# Success!
		$loggedin = 1;
		$serverPID = $rcvdData;
		print "You are logged in.\n";
		print PIDFILE "$username \t($password)\n";
	};
};

#printRoom($pos);
if (not $loggedin) {
	msgctl($cmqid, IPC_RMID, 0);
	close PIDFILE;
	unlink $pidfile;
	exit;
};
print "\n> ";


##############################################
#############  INIT ENDS HERE.  ##############
##############################################

# GO!
$loop->run;

msgctl($cmqid, IPC_RMID, 0);
close PIDFILE;
unlink $pidfile;
exit;

sub consoleInput {
	my ( $self, $buffref, $eof ) = @_;
	# Reset inactivity counter:
	$inact_timer = time;
	while( $$buffref =~ s/^(.*)\n// ) {
#		print "DEBUG: 1 is |$1|\n";
		my ($cmd, $args) = split(/ /, $1, 2);	
		if (not(defined($cmd))) { $cmd = '' };
		if (not(defined($args))) { $args = '' };
		$cmd =~ s/^\s*(.*?)\s*$/$1/;
		$cmd = uc($cmd);
		$args =~ s/^\s*(.*?)\s*$/$1/;
#		print "DEBUG: cmd is |$cmd|\n";
#		print "DEBUG: args is |$args|\n";
		if ($cmd eq 'LOGOUT') {
			my $message = "$clientPID cmdLogout";
			my $mqmsg = pack("l! a*", $mqmsgtype, $message);
			msgsnd($mmqid, $mqmsg, IPC_NOWAIT);
			$killrsp = kill('USR1', $serverPID);
			if (not ($killrsp)) {
				print "No response from server!\n";
				msgctl($cmqid, IPC_RMID, 0);
				close PIDFILE;
				unlink $pidfile;
				exit 0;
			};
		} elsif ($cmd eq '') {
			print "\r> ";
		} else {
			my $message = "$clientPID $cmd $args";
			my $bytes = do {use bytes; length($message)};
			if ($bytes > 2000) {
				$message = substr $message, 0, 2000;
				print "Message truncated!\n";
			};
			my $mqmsg = pack("l! a*", $mqmsgtype, $message);
			msgsnd($mmqid, $mqmsg, IPC_NOWAIT);
			$killrsp = kill('USR1', $serverPID);
			if (not ($killrsp)) {
				print "Server looks unreachable!\n";
				msgctl($cmqid, IPC_RMID, 0);
				close PIDFILE;
				unlink $pidfile;
				exit 0;
			};
		};
		print "\n> ";
	};
};

sub mainMudaTick { 
	#$timecntr++ ;
	# Placeholder:
	#print "$timecntr seconds have passed\n";
	# Read queue:
	my $buf;
	while (msgrcv($cmqid, $buf, 2048, 0, IPC_NOWAIT)){
		my $bytes = do {use bytes; length($buf)};
                my ($typ_rcvd, $rcvdData) = unpack("l! a*", $buf);
		# Check for server2client command:
		if ($rcvdData =~ /^SCMD2CLNT/) {
			# Verify pid:
			my ($rest, $ppid, $scmd) = split / /, $rcvdData, 3;
			if ($ppid == $clientPID) {
				if ($scmd eq 'EXIT') {
						print "\nDisconnecting...\n";
						msgctl($cmqid, IPC_RMID, 0);
						close PIDFILE;
						unlink $pidfile;
						exit;
				} else { 
						print "\nUnknown command from server...\n"
				};
			};
		} else {
                	print "$rcvdData";
		};
	};
	# Ping server:
	my $message = "$clientPID cmdPing";
	my $mqmsg = pack("l! a*", $mqmsgtype, $message);
	msgsnd($mmqid, $mqmsg, IPC_NOWAIT);
};

sub checkAlive {
	my $s = IO::Select->new();
	$s->add(\*STDOUT);
	my @ready = $s->can_write(1);
#	print "DEBUG: @ready\n";
	if (scalar(@ready) < 1) {
		# Lost connection:
		msgctl($cmqid, IPC_RMID, 0);
		close PIDFILE;
		unlink $pidfile;
		exit 0;
	};
	my $newtime = time;
	my $tt = $newtime - $inact_timer;
	my $t2 = $inact_logoff - $tt;
	if ($tt < $inact_warn) {
		# Just continue.
	} elsif ($tt < $inact_logoff) {
		# Warn about logoff.
		my $msg = "\rWarning: inactivity detected - auto-freeze imminent ($t2 seconds)!\n> ";
		print $msg;
	} else {
		my $msg = "\rInactivity detected ($tt seconds) - auto-freezing.\n";
		print $msg;
		my $message = "$clientPID cmdLogout";
		my $mqmsg = pack("l! a*", $mqmsgtype, $message);
		msgsnd($mmqid, $mqmsg, IPC_NOWAIT);
		$killrsp = kill('USR1', $serverPID);
		if (not ($killrsp)) {
			print "No response from server!\n";
			# Lost connection 
			msgctl($cmqid, IPC_RMID, 0);
			close PIDFILE;
			unlink $pidfile;
			exit 0;
		};
		print "\nDisconnecting...\n";
		msgctl($cmqid, IPC_RMID, 0);
		close PIDFILE;
		unlink $pidfile;
		exit 0;
		# Reset inactivity counter to avoid sending the logoff msg twice.
		$inact_timer = time;
	};
};
