#############################################################################
# MUDA: Resurrection							    #
# Ms_Subs.pm		: Various internal subroutines.			    # 
# Artist:		: Theodore J. Soldatos				    #
#############################################################################
# This file is part of MUDA:Resurrection.			   	    #
#									    #
# MUDA:Resurrection is free software: you can redistribute it and/or modify #
# it under the terms of the GNU General Public License as published by	    #
# the Free Software Foundation, either version 3 of the License, or	    #
# (at your option) any later version.					    #
#									    #
# MUDA:Resurrection is distributed in the hope that it will be useful,	    #
# but WITHOUT ANY WARRANTY; without even the implied warranty of	    #
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the		    #
# GNU General Public License for more details.				    #
#									    #
# You should have received a copy of the GNU General Public License	    #
# along with MUDA:Resurrection.  If not, see <http://www.gnu.org/licenses/>.#
#									    #
# Copyright 2012 Theodore J. Soldatos					    #
#############################################################################



package Ms_Libs;

use strict;
use warnings;
use DateTime::Format::SQLite;

use IPC::SysV qw(IPC_PRIVATE IPC_RMID IPC_CREAT S_IRWXU IPC_NOWAIT);
use Text::Autoformat;

BEGIN {
use Exporter ();
our ($VERSION, @ISA, @EXPORT, @EXPORT_OK, %EXPORT_TAGS);
# set the version for version checking
$VERSION = 1.00;
@ISA = qw(Exporter);
@EXPORT = qw(&logPrint &clientTell &tellAllUsers &clientLogout &clientPing &clientTimeouts &clientPrintLong &clientPrintRoom &informRoom &movePlayer &intPid2Details &intWearTakeoff &intWieldUnwield &intCreateItemFromProto &intDestroyItem &intLoginPage &intCalcHit &intHitInfo &userTell &npcTell &intCreateNpcFromProto &intUid2Details &npcRevival &intCmpUid2name &createFight &randomWalk &intTell &intDoOther &intCreateMoney &intMissileMove &intAdvExp &cmpItems &intCreateItemFromItem &intInvSelectNum &intConsolidateInv &defAlignement);
%EXPORT_TAGS = ( ); # eg: TAG => [ qw!name1 name2! ],
#@EXPORT_OK = qw();
}
our @EXPORT_OK;

sub logPrint {
	# For now, just print:
	my $tstamp = DateTime::Format::SQLite->format_datetime(DateTime->now);
	$tstamp =~ s/^(.*)\n//;
	my ($msg, undef) = @_;
	print "$tstamp $msg";
};

sub clientTell {
	# Send a message to the client on a specific queue:
	my ($clientmessageqid, $username, $clientPID, $message) = @_;
	if ($clientPID < 1) {
		# Avoid to signal all our processes. PID is -1 on 
		# frozen users, so just return.
		&logPrint("Negative pid $clientPID in clientTell sub.\n");
		return;
	};
	if (not defined ($main::pid2username_ref->{$clientPID})) {
		&logPrint("Unknown pid $clientPID in clientTell sub.\n");
		# Discard message to avoid server crash.
		# Try to kick out zombie client:
		kill('INT', $clientPID);
		return;
	};
	# Don't send more than 2K to the queue:
	my $bytes = do {use bytes; length($message)};
	if ($bytes > 2000) {
		$message = substr $message, 0, 1990;
		$message = "$message |TRUNC";
	};
	my $mqmsg = pack("l! a*", 1, "\r$message" . "$main::prompt ");
	if (not(msgsnd($clientmessageqid, $mqmsg, IPC_NOWAIT))) {
		&logPrint("Error trying to send message to $username ($clientPID)");
		&clientLogout($clientPID, 1);
	};
	$main::killrsp = kill('USR1', $clientPID);
	if (not ($main::killrsp)) {
		&logPrint("Error trying to SIGUSR1 $username ($clientPID)\n");
		&clientLogout($clientPID, 1);
	};
};

sub userTell {
	# Like clientTell, but checks if user is NPC and programmable.
	# Should be used instead of clientTell in any case where the user may be NPC.
	# This exists to avoid checking for NPCs. In some cases, it is not 
	# needed because we already know that the user is real, so we use 
	# clientTell with less overhead.
	my ($clientmessageqid, $username, $clientPID, $clientUID, $message) = @_;
	if (($clientmessageqid > 0) and ($clientPID > 0)) {
		# Real person, send message.
		&clientTell($clientmessageqid, $username, $clientPID, $message);
		return;
	} else {
		# NPC. If npc_code_ref exists, call it, otherwise discard message.
		my $npc_code = $main::usersMainData[ $clientUID ]->{ npc_code };
		my $npc_code_ref = $main::usersMainData[ $clientUID ]->{ npc_code_ref };
		my $validCode = $main::usersMainData[ $clientUID ]->{ validCode };
		if (($npc_code_ref ne '') and ($validCode == 1)) {
			no strict "refs";
			# Load code:
			#delete $INC{ $npc_code };
			#my $code = "require \'" . $npc_code . "\';";
			#eval $code;
			#if ($@) {
			#	warn $@;
			#		# Can't do much without code, return.
			#		return;
			#};
			# Call routine with required param __INFO. 
			&{ $npc_code_ref }(-1, $clientUID, '__INFO', $message);
			return;
		} else {
			# No code to handle that, return.
			return;
		};
	};
};

sub npcTell {
	# A more generic version of userTell that calls arbitrary routines of the NPC.
	# Also more specific, in the sense that is used only on NPCs.
	my ($c_pid, $uid, $npcCmd, $args) = @_;
	my $npc_code = $main::usersMainData[ $uid ]->{ npc_code };
	my $npc_code_ref = $main::usersMainData[ $uid ]->{ npc_code_ref };
	my $validCode = $main::usersMainData[ $uid ]->{ validCode };
	if (($npc_code_ref ne '') and ($validCode == 1)) {
		no strict "refs";
		# Load code:
		#delete $INC{ $npc_code };
		#my $code = "require \'" . $npc_code . "\';";
		#eval $code;
		#if ($@) {
		#	warn $@;
		#	# Can't do much without code, return.
		#	return;
		#};
		# Call routine. 
		my $rc = &{ $npc_code_ref }($c_pid, $uid, $npcCmd, $args);
		return $rc;
	} else {
		# No code to handle that, return.
		return;
	};
};

sub tellAllUsers {
	# Go through all logged in users and send them a message:
	my ($msg, undef) = @_;
	foreach my $username (@main::loggedInUsers) {
		&clientTell($main::usersMainData[ $main::userPass_ref->{$username}->{uid} ]->{ qid }, $username, $main::usersMainData[ $main::userPass_ref->{$username}->{uid} ]->{ pid }, $msg);
	};
};

sub clientTimeouts {
	foreach my $username (@main::loggedInUsers) {
		my $uid = $main::userPass_ref->{ $username }->{ uid };
		if (++$main::usersMainData[ $uid ]->{ timeout } > $main::clientTimeout) {
			my $pid = $main::usersMainData[ $uid ]->{ pid };
			&clientLogout($pid ,1);
		};
	};
};

sub clientPing {
	my ($pid, undef) = @_;
	my ($username, $uid, $cmqid, $roomNum, $gender) = &intPid2Details($pid);
	# Border case:
	if (not defined $uid) {return;};
	# Just reset inactivity counter:
	$main::usersMainData[ $uid ]->{ timeout } = 0;
};

sub clientLogout {
	my ($pid, $auto, undef) = @_;
	my ($username, $uid, $cmqid, $roomNum, $gender) = &intPid2Details($pid);
	$main::usersMainData[ $uid ]->{ lastDiscon } = DateTime::Format::SQLite->format_datetime(DateTime->now);
	if (not defined $auto) { $auto = 0;};
	if (not defined ($main::pid2username_ref->{$pid})) {
		&logPrint("Unknown pid $pid in logout sub.\n");
		# Try to kick out zombie client:
		kill('INT', $pid);
	} else {
		# Save first:
		my $username = $main::pid2username_ref->{$pid};
		my $cmqid = $main::usersMainData[ $uid ]->{ qid };
		my $roomNum = $main::usersMainData[$uid]->{ room };
		use DBD::SQLite;
		my $dbh = DBI->connect("dbi:SQLite:dbname=$main::dbfile","",""); 
		&saveSingleUser($dbh, $uid);
		$dbh->disconnect;
		@main::loggedInUsers = grep(!/$username/, @main::loggedInUsers);
		$main::usersMainData[ $uid ]->{ pid } = -1;
		$main::usersMainData[ $uid ]->{ qid } = -1;
		$main::usersMainData[ $uid ]->{ timeout } = 0;
		delete $main::pid2username_ref->{$pid};
		$auto || &clientTell($cmqid, $username, $pid, 'You are logged out.\n');
		my $msg = "SCMD2CLNT $pid EXIT";
		my $mqmsg = pack("l! a*", 1, $msg);
		#$auto || &clientTell($cmqid, $username, $pid, $msg);
		$auto || msgsnd($cmqid, $mqmsg, IPC_NOWAIT);
		&tellAllUsers("$username disconnected.\n");
		&informRoom($roomNum, $uid, "$username turns into an ice statue!\n");
		$auto || &logPrint("$username ($pid) logged off.\n");
		$auto && &logPrint("$username ($pid) automatically freezes.\n");
		# Try to remove client's msg queue:
		$auto && msgctl($cmqid, IPC_RMID, 0);
	};
};

sub clientPrintLong {
	my ($cmqid, $username, $pid, $line, undef) = @_;
	my $formatted = autoformat $line, { left=>1, right=>$main::termWidth, justify => 'full' };
	&clientTell($cmqid, $username, $pid, $formatted);
};

sub clientPrintRoom {
	# Print current room:
	my ($cmqid, $username, $pid, undef) = @_;
	my (undef, $uid, undef, $roomNum, $gender) = &intPid2Details($pid);

	&clientTell($cmqid, $username, $pid, "Room #$roomNum\n\n");
	&clientPrintLong($cmqid, $username, $pid, $main::roomsLongDescr[$roomNum]);
	&clientTell($cmqid, $username, $pid, "Exits:\n");
	my $msg = '';
	$msg = $msg . "\tEast:\t$main::roomsShortDescr[ $main::roomsData[$roomNum]{exE} ]\n" if ($main::roomsData[$roomNum]{exE} != 0);
	$msg = $msg . "\tWest:\t$main::roomsShortDescr[ $main::roomsData[$roomNum]{exW} ]\n" if ($main::roomsData[$roomNum]{exW} != 0);
	$msg = $msg . "\tNorth:\t$main::roomsShortDescr[ $main::roomsData[$roomNum]{exN} ]\n" if ($main::roomsData[$roomNum]{exN} != 0);
	$msg = $msg . "\tSouth:\t$main::roomsShortDescr[ $main::roomsData[$roomNum]{exS} ]\n" if ($main::roomsData[$roomNum]{exS} != 0);
	$msg = $msg . "\tUp:\t$main::roomsShortDescr[ $main::roomsData[$roomNum]{exU} ]\n" if ($main::roomsData[$roomNum]{exU} != 0);
	$msg = $msg . "\tDown:\t$main::roomsShortDescr[ $main::roomsData[$roomNum]{exD} ]\n" if ($main::roomsData[$roomNum]{exD} != 0);
	$msg = $msg . "\tNE:\t$main::roomsShortDescr[ $main::roomsData[$roomNum]{exNE} ]\n" if ($main::roomsData[$roomNum]{exNE} != 0);
	$msg = $msg . "\tNW:\t$main::roomsShortDescr[ $main::roomsData[$roomNum]{exNW} ]\n" if ($main::roomsData[$roomNum]{exNW} != 0);
	$msg = $msg . "\tSE:\t$main::roomsShortDescr[ $main::roomsData[$roomNum]{exSE} ]\n" if ($main::roomsData[$roomNum]{exSE} != 0);
	$msg = $msg . "\tSW:\t$main::roomsShortDescr[ $main::roomsData[$roomNum]{exSW} ]\n" if ($main::roomsData[$roomNum]{exSW} != 0);
	&clientTell($cmqid, $username, $pid, $msg);
};

sub informRoom {
	my ($roomNum, $excludeUid, $msg, $excludeMore, undef) = @_;
	# First inform real people:
	if (not defined $excludeUid) { $excludeUid = -1 };
	if (not defined $excludeMore) { $excludeMore = -1 };
	if (scalar(@{ $main::roomsData[ $roomNum ]->{presentUids} }) > 0) {
		# There are people here.
		foreach my $ouid (@{ $main::roomsData[ $roomNum ]->{presentUids} }) {
			if (($ouid == $excludeUid) or ($ouid == $excludeMore)) {next;};
			if ($main::usersMainData[$ouid]->{ pid } < 0) { next;};
			if ($main::usersMainData[$ouid]->{ npc } == 1) { next;};
			my $cmqid = $main::usersMainData[ $ouid ]->{ qid };
			my $pid = $main::usersMainData[ $ouid ]->{ pid };
			my $username = $main::pid2username_ref->{$pid};
			&clientTell($cmqid, $username, $pid, $msg);
		};
		# Then inform NPCs. We avoid to use userTell here, instead, we load and call 
		# each NPC.
		foreach my $ouid (@{ $main::roomsData[ $roomNum ]->{presentUids} }) {
			if (($ouid == $excludeUid) or ($ouid == $excludeMore)) {next;};
			if ($main::usersMainData[$ouid]->{ npc } == 0) { next;};
			if ($main::usersMainData[$ouid]->{ npc_code } eq '') { next;};
			if ($main::usersMainData[$ouid]->{ npc_code_ref } eq '') { next;};
			if ($main::usersMainData[$ouid]->{ validCode } == 0) { next;};
			no strict "refs";
			# Load code:
			# Code is reevaluated every time and not used in the case of syntax errors.
			#my $npc_code = $main::usersMainData[$ouid]->{ npc_code };
			my $npc_code_ref = $main::usersMainData[$ouid]->{ npc_code_ref };
			#delete $INC{ $npc_code };
			#my $code = "require \'" . $npc_code . "\';";
			#eval $code;
			#if ($@) {
			#	warn $@;
			#	# Can't do much without code, return.
			#	return;
			#};
			# Call routine with required param __INFO. 
			&{ $npc_code_ref }(-1, $ouid, '__INFO', $msg);
		};
	};
	# Then inform items:
	if (scalar(keys %{ $main::roomsData[ $roomNum ]->{presentItems} }) > 0) {
		# Check which of these have code:
		ALLIT:	foreach my $iid (keys %{ $main::roomsData[ $roomNum ]->{presentItems} }) {
			my $item_ref = $main::roomsData[ $roomNum ]->{presentItems}->{$iid};
			if (
				(defined $item_ref->{ executable }) and 
				($item_ref->{ executable } ne '') and 
				(defined $item_ref->{ code_ref }) and
				($item_ref->{ code_ref } ne '') and
				(defined $item_ref->{ validCode }) and
				($item_ref->{ validCode } == 1)
			) {
				no strict "refs";
				# Load code:
				#delete $INC{ $item_ref->{ executable } };
				#my $code = "require \'" . $item_ref->{ executable } . "\';";
				#eval $code;
				#if ($@) {
				#	warn $@;
				#	next ALLIT;
				#};
				# Now run the code with message as parameter:
				my $rc = &{ $item_ref->{ code_ref } }($main::usersMainData[ $excludeUid ]->{ pid }, '__INFORM', $iid, $msg);
			};
		};
	};
};

sub movePlayer {
	# Move $pid from current pos to $newpos.
	my ($uid, $newpos, undef) = @_;
	my ($username, $pid, $cmqid, $roomNum, $gender) = &intUid2Details($uid);
	# Update user data:
	$main::usersMainData[$uid]->{ room } = $newpos;
	# Remove player from old room:
	my( $index )= grep { ${ $main::roomsData[ $roomNum ]->{ presentUids } }[$_] eq $uid } 0..$#{ $main::roomsData[ $roomNum ]->{ presentUids } };
	splice(@{ $main::roomsData[ $roomNum ]->{ presentUids } }, $index, 1);
	my( $indexu )= grep { ${ $main::roomsData[ $roomNum ]->{ presentUsers } }[$_] eq $username } 0..$#{ $main::roomsData[ $roomNum ]->{ presentUsers } };
	splice(@{ $main::roomsData[ $roomNum ]->{ presentUsers } }, $indexu, 1);
	# Add player in the new room:
	push @{ $main::roomsData[ $newpos ]->{presentUids} }, $uid;
	push @{ $main::roomsData[ $newpos ]->{presentUsers} }, $username;
};

sub intPid2Details {
	# Find often used user details from pid and return them.
	# This works only for logged in users.
	my ($pid, undef) = @_;
	my @retArr = ();
	if (not defined $main::pid2username_ref->{$pid}) {
		return undef;
	};
	my $username = $main::pid2username_ref->{$pid};
	my $uid = $main::userPass_ref->{ $username }->{ uid };
	my $cmqid = $main::usersMainData[ $uid ]->{ qid };
	my $roomNum = $main::usersMainData[$uid]->{ room };
	my $gender = $main::usersMainData[$uid]->{ gender };
	push(@retArr, $username, $uid, $cmqid, $roomNum, $gender);
	return @retArr;
};

sub intUid2Details {
	# Find often used user details from uid and return them.
	my ($uid, undef) = @_;
	my @retArr = ();
	if (not defined $main::usersMainData[ $uid ]->{ username }) {
		return undef;
	};
	my $username = $main::usersMainData[ $uid ]->{ username };
	my $cmqid = $main::usersMainData[ $uid ]->{ qid };
	my $pid = $main::usersMainData[ $uid ]->{ pid };
	my $roomNum = $main::usersMainData[$uid]->{ room };
	my $gender = $main::usersMainData[$uid]->{ gender };
	push(@retArr, $username, $pid, $cmqid, $roomNum, $gender);
	return @retArr;
};

sub intWearTakeoff {
	# Wear and take off item.
	my ($pid, $intCmd, $args) = @_;
	# $intCmd: WE, TO 
	my ($username, $uid, $cmqid, $roomNum, $gender) = &intPid2Details($pid);
	# Scan inventory for item:
	my $inv_ref = $main::usersMainData[ $uid ]->{ inventory };
	my @items = ();
	if ($args =~ /^#/) {
		# Special syntax with internal item id to resolve ambiguities
		my ($scrap, $iid) = split /#/, $args, 2;
		if (not defined($inv_ref->{ $iid })) {
			&clientTell($cmqid, $username, $pid, "Item #$iid invalid.\n");
			return;
		};
		if (
			($inv_ref->{ $iid }->{ wearable } == 0) or 
			($inv_ref->{ $iid }->{ userID } == 0) or
			($inv_ref->{ $iid }->{ userID } != $uid)
		) {
			&clientTell($cmqid, $username, $pid, "Item #$iid invalid.\n");
			return;
		};
		push @items, $iid;
	} else {
		$args = quotemeta($args);
		foreach my $iid (keys %$inv_ref) {
			my $shortDesc = $inv_ref->{ $iid }->{ shortDesc };
			if ($shortDesc =~ m/$args/i) {
				if ((
					($inv_ref->{ $iid }->{ worn } == 1) and 
					($intCmd eq 'TO')
				) or ( 
					($inv_ref->{ $iid }->{ worn } == 0) and 
					($intCmd eq 'WE')
				)) {
					push @items, $iid;
				};
			};
		};
	};	
	if (scalar(@items) == 0) {
		# Not found.
		&clientTell($cmqid, $username, $pid, "You have not any $args.\n");
	} elsif (scalar(@items) > 1) {
		# Found more than one:
		&clientTell($cmqid, $username, $pid, "Please be more specific. Which one?\n");
		foreach my $iid (@items) {
			&clientTell($cmqid, $username, $pid, "#$iid\t: " . $inv_ref->{ $iid }->{ shortDesc } . ".\n");
		};
	} else {
		# Exactly one: 
		my $iid = pop @items;
		my $rec = $main::usersMainData[ $uid ]->{ inventory }->{ $iid };

		if (($rec->{ wearable } == 0) and ($intCmd eq 'WE')) {
			&clientTell($cmqid, $username, $pid, "You can't wear that!\n");
			return;
		};
		if (($rec->{ worn } == 0) and ($intCmd eq 'TO')) {
			&clientTell($cmqid, $username, $pid, "You don't wear any $args!\n");
			return;
		};
		my $itemType = $rec->{ gen_item_type };
		if ($intCmd eq 'WE') {
			# Count worn items of the same type:
			my $lcnt = 0;
			foreach my $iid (keys %$inv_ref) {
				if (($inv_ref->{ $iid }->{ gen_item_type } == $itemType) and 
					($inv_ref->{ $iid }->{ worn } == 1)) {
					$lcnt++;
				};
			};
			if ($lcnt >= $main::itemTypes{ $itemType }->{ wwLimit }) {
				# You can't wear two hats.
				&clientTell($cmqid, $username, $pid, "You can't wear " . $rec->{ shortDesc } . ". Take something off first!\n");
				return;
			};
		};
		if ($intCmd eq 'WE') {
			# If item is multiple, decrease amount, create a new one and use the
			# new iid for the rest of the command.
			if ($main::itemData{ $iid }->{ amount } > 1) {
				my $niid = &intCreateItemFromItem($pid, $iid, 1);
				if (not $niid) {
					# Error:
					&clientTell($cmqid, $username, $pid, "You failed to wear $rec->{ shortDesc } (internal error).\n");
					return;
				};
				# New item created, decrease amount and work with new item:
				$main::itemData{ $iid }->{ amount } -= 1;
				$main::itemData{ $niid }->{ amount } = 1;
				$iid = $niid;
			};
			# Change main data structure:
			$main::itemData{ $iid }->{ worn } = 1;
			# Change inventory data structure: 
			$main::usersMainData[ $uid ]->{ inventory }->{ $iid }->{ worn } = 1;
			&clientTell($cmqid, $username, $pid, "You are now wearing " . $rec->{ shortDesc } . ".\n");
			if ($main::usersMainData[$uid]->{ invisible } == 0) {
				&informRoom($roomNum, $uid, "$username wears " . $rec->{ shortDesc } . "\n");
			};
		} elsif ($intCmd eq 'TO') {
			# Change main data structure:
			$main::itemData{ $iid }->{ worn } = 0;
			# Change inventory data structure: 
			$rec->{ worn } = 0;
			&clientTell($cmqid, $username, $pid, "You take off " . $rec->{ shortDesc } . ".\n");
			if ($main::usersMainData[$uid]->{ invisible } == 0) {
				&informRoom($roomNum, $uid, "$username takes off " . $rec->{ shortDesc } . "\n");
			};
		} else {
			# Should not get here:
			&logPrint("Fall off in intWearTakeoff\n"); 
			&clientTell($cmqid, $username, $pid, "System failure.\n");
		};
	};
};


sub intWieldUnwield {
	# Wield and Unwield item.
	my ($pid, $intCmd, $args) = @_;
	# $intCmd: WI, UN 
	my ($username, $uid, $cmqid, $roomNum, $gender) = &intPid2Details($pid);
	# Scan inventory for item:
	my $inv_ref = $main::usersMainData[ $uid ]->{ inventory };
	my @items = ();
	if ($args =~ /^#/) {
		# Special syntax with internal item id to resolve ambiguities
		my ($scrap, $iid) = split /#/, $args, 2;
		if (not defined($inv_ref->{ $iid })) {
			&clientTell($cmqid, $username, $pid, "Item #$iid invalid.\n");
			return;
		};
		if (
			($inv_ref->{ $iid }->{ wieldable } == 0) or 
			($inv_ref->{ $iid }->{ userID } == 0) or
			($inv_ref->{ $iid }->{ userID } != $uid)
		) {
			&clientTell($cmqid, $username, $pid, "Item #$iid invalid.\n");
			return;
		};
		push @items, $iid;
	} else {
		$args = quotemeta($args);
		foreach my $iid (keys %$inv_ref) {
			my $shortDesc = $inv_ref->{ $iid }->{ shortDesc };
			if ($shortDesc =~ m/$args/i) {
				if ((
					($inv_ref->{ $iid }->{ wielded } == 1) and 
					($intCmd eq 'UN')
				) or ( 
					($inv_ref->{ $iid }->{ wielded } == 0) and 
					($intCmd eq 'WI')
				)) {
					push @items, $iid;
				};
			};
		};
	};
	if (scalar(@items) == 0) {
		# Not found.
		&clientTell($cmqid, $username, $pid, "You have not any $args.\n");
	} elsif (scalar(@items) > 1) {
		# Found more than one:
		&clientTell($cmqid, $username, $pid, "Please be more specific. Which one?\n");
		foreach my $iid (@items) {
			&clientTell($cmqid, $username, $pid, "#$iid\t:" . $inv_ref->{ $iid }->{ shortDesc } . ".\n");
		};
	} else {
		# Exactly one: 
		my $iid = pop @items;
		my $rec = $main::usersMainData[ $uid ]->{ inventory }->{ $iid };

		if (($rec->{ wieldable } == 0) and ($intCmd eq 'WI')) {
			&clientTell($cmqid, $username, $pid, "You can't wield that!\n");
			return;
		};
		if (($rec->{ wielded } == 0) and ($intCmd eq 'UN')) {
			&clientTell($cmqid, $username, $pid, "You don't wield any $args!\n");
			return;
		};
		my $itemType = $rec->{ gen_item_type };
		if ($intCmd eq 'WI') {
			# Count wielded items of the same type:
			my $lcnt = 0;
			foreach my $iid (keys %$inv_ref) {
				if (($inv_ref->{ $iid }->{ gen_item_type } == $itemType) and 
					($inv_ref->{ $iid }->{ wielded } == 1)) {
					$lcnt++;
				};
			};
			if ($lcnt >= $main::itemTypes{ $itemType }->{ wwLimit }) {
				&clientTell($cmqid, $username, $pid, "You can't wield " . $rec->{ shortDesc } . ". Unwield something first!\n");
				return;
			};
		};
		if ($intCmd eq 'WI') {

			# If item is multiple, decrease amount, create a new one and use the
			# new iid for the rest of the command.
			if ($main::itemData{ $iid }->{ amount } > 1) {
				my $niid = &intCreateItemFromItem($pid, $iid, 1);
				if (not $niid) {
					# Error:
					&clientTell($cmqid, $username, $pid, "You failed to wield $rec->{ shortDesc } (internal error).\n");
					return;
				};
				# New item created, decrease amount and work with new item:
				$main::itemData{ $iid }->{ amount } -= 1;
				$main::itemData{ $niid }->{ amount } = 1;
				$iid = $niid;
			};

			# Change main data structure:
			$main::itemData{ $iid }->{ wielded } = 1;
			# Change inventory data structure: 
			$main::usersMainData[ $uid ]->{ inventory }->{ $iid }->{ wielded } = 1;
			&clientTell($cmqid, $username, $pid, "You are now wielding " . $rec->{ shortDesc } . ".\n");
			if ($main::usersMainData[$uid]->{ invisible } == 0) {
				&informRoom($roomNum, $uid, "$username wields " . $rec->{ shortDesc } . "\n");
			};
		} elsif ($intCmd eq 'UN') {
			# Change main data structure:
			$main::itemData{ $iid }->{ wielded } = 0;
			# Change inventory data structure: 
			$rec->{ wielded } = 0;
			&clientTell($cmqid, $username, $pid, "You unwield " . $rec->{ shortDesc } . ".\n");
			if ($main::usersMainData[$uid]->{ invisible } == 0) {
				&informRoom($roomNum, $uid, "$username unwields " . $rec->{ shortDesc } . "\n");
			};
		} else {
			# Should not get here:
			&logPrint("Fall off in intWieldUnwield\n"); 
			&clientTell($cmqid, $username, $pid, "System failure.\n");
		};
	};
};

sub intCreateMoney {
	# Create a coin item of a specific amount in a specific room.
	my ($roomId, $amount, undef) = @_;
	if ($amount <= 0) {
		# Fail.
		return 0;
	};
	# Prototype 7 is coins.
	my $iid = &intCreateItemFromProto($roomId, 7, 0);
	if ($iid <= 0) {
		# Failed.
		return 0;
	};
	# Item created, fix descriptions and amount.
	if ($amount > 1) {
		# Default item is 1 coin, alter it only when > 1.
		$main::itemData{ $iid }->{ amount } = $amount;
		$main::itemData{ $iid }->{ shortDesc } = "$amount gold coins";
		$main::itemData{ $iid }->{ longDesc } = "$amount gold coins";
	};
	# Load item code:
	if ($main::itemData{$iid}->{ code_ref } ne '') {
		no strict "refs";
		# Load new code:
		my $executable = $main::itemData{$iid}->{ executable };
		delete $INC{ $executable };
		my $code = "require '$executable'";
		eval $code;
		if ($@) {
			&logPrint("Command code error for item #$iid: $@\n");
			$main::itemData{ $iid }->{ validCode } = 0;
		} else {
			$main::itemData{ $iid }->{ validCode } = 1;
		};
	};

	return $iid;
};

sub intCreateItemFromProto {
	my ($pid, $piid, $isUser, $args) = @_;
	# First argument ($pid) is roomNum when not $isUser and uid when $isUser = 2 (NPC).
	if ($main::itemLock == 1) { 
		&logPrint("Failed createItemFromProto: lock\n");
		return 0; 
	};
	$main::itemLock = 1;
	my ($username, $uid, $cmqid, $roomNum, $gender);
	if ($isUser == 1) {
		($username, $uid, $cmqid, $roomNum, $gender) = &intPid2Details($pid);
	};
	if ($isUser == 2) {
		$uid = $pid;
		($username, $pid, $cmqid, $roomNum, $gender) = &intUid2Details($uid);
	};
	# Copy prototype record:
	my %newItem = %{ $main::itemProto{ $piid } };
	# Add missing keys
	if ($isUser) {
		$newItem{ roomID } = 0;
		$newItem{ userID } = $uid;
	} else {
		# Is room
		$newItem{ roomID } = $pid;
		$newItem{ userID } = 0;
	};
	$newItem{ itemID } = 0;
	$newItem{ worn } = 0;
	$newItem{ wielded } = 0;
	$newItem{ validCode } = 0;
	# Define new item id:
	my $niid = $main::maxItemId + 1;
	# Do a redundant check of existance:
	if (defined $main::itemData{ $niid }) {
		# Somehow it exists. Increase $maxItemId and return fail.
		&logPrint("Failed createItemFromProto: exists\n");
		$main::maxItemId++;
		return 0;
	};
	$newItem{ id } = $niid;
	# Create new item:
	# If needed, add notifier in main loop:
	if (
		($newItem{ loop_ref } ne '') and 
		($newItem{ loop_secs } > 0)
	) {
		no strict "refs";
		# Load code:
		my $code = "require '" . $newItem{ executable } . "'";
		eval $code;
		if ($@) {
			&logPrint("Notifier error for item #$niid: $@\n");
		} else {
			my $notifier_ref = IO::Async::Timer::Periodic->new(
				interval 	=>	$newItem{ loop_secs },
				notifier_name	=>	"ITEM_$niid",
				on_tick		=>	\&{ $newItem{ loop_ref } }
			);
			$notifier_ref->start;
			$main::loop->add($notifier_ref);
			$newItem{ validCode } = 1;
		};
	} elsif ($newItem{ code_ref } ne '') {
		# Load code:
		my $code = "require '" . $newItem{ executable } . "'";
		eval $code;
		if ($@) {
			&logPrint("Code error for item #$niid: $@\n");
		} else {
			$newItem{ validCode } = 1;
		};
	};
	$main::itemData{ $niid } = \%newItem;
	if ($isUser) {
		$main::usersMainData[ $uid ]->{ inventory }->{ $niid } = \%newItem;
	} else {
		$main::roomsData[ $pid ]->{ presentItems }->{ $niid } = \%newItem;
	};
	$main::maxItemId = $niid;
	$main::itemLock = 0;
	return $niid;
};

sub intDestroyItem {
	my ($piid, $args) = @_;
	# See if exists:
	if (defined $main::itemData{ $piid }) {
		# Get room or user data:
		my $roomNum = $main::itemData{ $piid }->{ roomID };
		my $uid = $main::itemData{ $piid }->{ userID };
		# Locate and remove item notifier for "smart" items.
		if ($main::itemData{ $piid }->{ loop_ref } ne '') {
			my $notifier_name = "ITEM_$piid";
			my @notifiers = $main::loop->notifiers;
			foreach my $notif (@notifiers) {
				if ($notif->notifier_name eq $notifier_name) {
					$main::loop->remove($notif);
				};
			};
		};

		if ($roomNum > 0) {
			# Delete entry from room data:
			delete $main::roomsData[ $roomNum ]->{ presentItems }->{ $piid };
		};
		if ($uid > 0) {
			# Delete entry from user data:
			delete $main::usersMainData[ $uid ]->{ inventory }->{ $piid };
		};
		# Delete entry from item data:
		delete $main::itemData{ $piid };
		return $piid;
	} else {
		return 0;
	};
};

sub intCreateItemFromItem {
	my ($pid, $piid, $isUser, $args) = @_;
	# First argument ($pid) is roomNum when not $isUser, uid when $isUser==2 (NPC).
	if ($main::itemLock == 1) { 
		&logPrint("Failed createItemFromItem: lock\n");
		return 0; 
	};
	$main::itemLock = 1;
	my ($username, $uid, $cmqid, $roomNum, $gender);
	if ($isUser == 1) {
		($username, $uid, $cmqid, $roomNum, $gender) = &intPid2Details($pid);
	};
	if ($isUser == 2) {
		$uid = $pid;
		($username, $pid, $cmqid, $roomNum, $gender) = &intUid2Details($pid);
	};
	# Copy prototype record:
	my %newItem = %{ $main::itemData{ $piid } };
	# Define new item id:
	my $niid = $main::maxItemId + 1;
	# Do a redundant check of existance:
	if (defined $main::itemData{ $niid }) {
		# Somehow it exists. Increase $maxItemId and return fail.
		&logPrint("Failed createItemFromItem: exists\n");
		$main::maxItemId++;
		return 0;
	};
	$newItem{ id } = $niid;
	# Create new item:
	# If needed, add notifier in main loop:
	if (($newItem{ loop_ref } ne '') and ($newItem{ loop_secs } > 0)) {
		no strict "refs";
		# Load code:
		my $code = "require '" . $newItem{ executable } . "'";
		eval $code;
		if ($@) {
			&logPrint("Notifier error for item #$niid: $@\n");
			$newItem{ validCode } = 1;
		} else {
			my $notifier_ref = IO::Async::Timer::Periodic->new(
				interval 	=>	$newItem{ loop_secs },
				notifier_name	=>	"ITEM_$niid",
				on_tick		=>	\&{ $newItem{ loop_ref } }
			);
			$notifier_ref->start;
			$main::loop->add($notifier_ref);
			$newItem{ validCode } = 1;
		};
	};
	$main::itemData{ $niid } = \%newItem;
	if ($isUser) {
		$main::usersMainData[ $uid ]->{ inventory }->{ $niid } = \%newItem;
	} else {
		$main::roomsData[ $pid ]->{ presentItems }->{ $niid } = \%newItem;
	};
	$main::maxItemId = $niid;
	$main::itemLock = 0;
	return $niid;
};


sub intLoginPage {
	my ($pid, $args) = @_;
	my ($username, $uid, $cmqid, $roomNum, $gender) = &intPid2Details($pid);
	my $msg;

	$msg = " _ _____________________________________________________________________";
	&clientTell($cmqid, $username, $pid, "$msg\n");
	$msg = "/ \\                     Welcome to MUDA:Resurrection!                    \\";
	&clientTell($cmqid, $username, $pid, "$msg\n");
	$msg = "|  |                    -----------------------------                     \\";
	&clientTell($cmqid, $username, $pid, "$msg\n");
	$msg = "\\__|                        Version $main::mrVersion                        |";
	&clientTell($cmqid, $username, $pid, "$msg\n");
	$msg = "   >----------------------------------------------------------------------<";
	&clientTell($cmqid, $username, $pid, "$msg\n");
	$msg = "   | Statistics:                                                          |";
	&clientTell($cmqid, $username, $pid, "$msg\n");
#	$msg = "   |                                                                      |";
#	&clientTell($cmqid, $username, $pid, "$msg\n");
	$msg = "   | Rooms: $main::stats_ref->{ roomCount }\tUsers: $main::stats_ref->{ userCount }\t\tNPCs: $main::stats_ref->{ npcCount }\t\t  |";
	&clientTell($cmqid, $username, $pid, "$msg\n");
#	$msg = "   |                                                                      |";
#	&clientTell($cmqid, $username, $pid, "$msg\n");
	$msg = "   | Items: $main::stats_ref->{ itemsCount }\t\tCoded: $main::stats_ref->{ codedItemsCount }\t\tActive:	$main::stats_ref->{ loopItemsCount }\t\t  |";
	&clientTell($cmqid, $username, $pid, "$msg\n");
#	$msg = "   |                                                                      |";
#	&clientTell($cmqid, $username, $pid, "$msg\n");
	$msg = "   | Proto: $main::stats_ref->{ itemProtoCount }\t\tCoded: $main::stats_ref->{ codedItemProtoCount }\t\tActive: $main::stats_ref->{ loopItemProtoCount }\t\t  |";
	&clientTell($cmqid, $username, $pid, "$msg\n");
#	$msg = "   |                                                                      |";
#	&clientTell($cmqid, $username, $pid, "$msg\n");
	$msg = "   | Item types: $main::stats_ref->{ itemTypes } \tQuests: $main::stats_ref->{ quests }\t\tCommands: $main::stats_ref->{ commands }\t\t  |";
	&clientTell($cmqid, $username, $pid, "$msg\n");
	$msg = "   |                                                                      |";
	&clientTell($cmqid, $username, $pid, "$msg\n");
	$msg = "   |                    Use HELP for some usefull info                    |";
	&clientTell($cmqid, $username, $pid, "$msg\n");
	$msg = "   |                                                                      |";
	&clientTell($cmqid, $username, $pid, "$msg\n");
	$msg = "   |    /---------------------------------------------------------------------/";
	&clientTell($cmqid, $username, $pid, "$msg\n");
	$msg = "   |   /                                                                     /";
	&clientTell($cmqid, $username, $pid, "$msg\n");
	$msg = "    \\_/_____________________________________________________________________/";
	&clientTell($cmqid, $username, $pid, "$msg\n");
};

sub intCalcHit {
	# Takes two uids (attacker/defender) and calculates damage.
	my ($attUid, $defUid, undef) = @_;
	# Define weapon power. By default, weapon pow is 0.5 (fists).
	my $attInv_ref = $main::usersMainData[ $attUid ]->{ inventory };
	my $attWC = 0.5;
	foreach my $iid (keys %$attInv_ref) {
		# Add all wielded weapons' WC, except bows.
		if (
			($attInv_ref->{ $iid}->{ wielded } == 1) and
			($attInv_ref->{ $iid}->{ missile_launcher } == 0) 
		) {
			$attWC = $attWC + $attInv_ref->{ $iid}->{ weaponClass };
		};
	};
	# Final result depends on attacker's health percentage
	$attWC = $attWC * ($main::usersMainData[ $attUid ]->{ hitPoints } / $main::usersMainData[ $attUid ]->{ maxHp });
	# A limit to avoid strange results:
	$attWC = 99 if $attWC > 99;
	$attWC = 0 if $attWC < 0;
	# Add 5% for each level > 1:
	$attWC = $attWC + (($attWC*0.05) * ($main::usersMainData[ $attUid ]->{ level } - 1));
	# Now define armour class of victim:
	my $defInv_ref = $main::usersMainData[ $defUid ]->{ inventory };
	my $defAC = 0;
	foreach my $iid (keys %$defInv_ref) {
		# Add all worn armour's AC.
		if ($defInv_ref->{ $iid}->{ worn } == 1) {
			$defAC = $defAC + $defInv_ref->{ $iid}->{ armourClass };
		};
	};
	# Add 5% for each level > 1:
	$defAC = $defAC + (($defAC*0.05) * ($main::usersMainData[ $defUid ]->{ level } - 1));
	# Total AC absorbs AC% of attWC, not damage.
	$attWC = $attWC - ($attWC * $defAC / 100);
	# Also substract or add a random percentage between -50% and +50%
	my $damage = $attWC + ($attWC * ((rand(100) - 50)/100));
	# If low enough, make it zero.
	#$damage = 0 if $damage < 1;
	# Triple it to make fights faster and less silly (less tickling).
	$damage *= 3; 
	# Round to two decimals:
	$damage = (sprintf("%.2f", $damage)) + 0;
	return $damage;
};

sub intHitInfo {
	# Displays message about hits. Based on original MUDA code.
	my ($attUid, $defUid, $damage, $roomNum) = @_;
	my $attcmqid = $main::usersMainData[ $attUid ]->{ qid };
	my $attpid = $main::usersMainData[ $attUid ]->{ pid };
	my $attusername = $main::usersMainData[ $attUid ]->{ username };
	my $defcmqid = $main::usersMainData[ $defUid ]->{ qid };
	my $defpid = $main::usersMainData[ $defUid ]->{ pid };
	my $defusername = $main::usersMainData[ $defUid ]->{ username };
	# Define message:
	my ($ToDef, $ToAtt, $ToOther);
	my $attusername2 = 'Somebody';
	if ($main::usersMainData[$attUid]->{ invisible } == 0) {
		$attusername2 = $attusername;
	};
	my $defusername2 = 'Somebody';
	if ($main::usersMainData[$defUid]->{ invisible } == 0) {
		$defusername2 = $defusername;
	};

	if ($damage <= 0) {
		$ToDef = "$attusername2 misses you.\n";
		$ToAtt = "You miss $defusername2.\n";
		$ToOther = "$attusername2 misses $defusername2.\n";
	} elsif (($damage > 0) and ($damage <= 5)) {
		$ToDef = "$attusername2 tickles you in the stomach.\n";
		$ToAtt = "You tickled $defusername2 in the stomach.\n";
		$ToOther = "$attusername2 tickles $defusername2 in the stomach.\n";
	} elsif (($damage > 5) and ($damage <= 10)) {
		$ToDef = "$attusername2 grazed you.\n";
		$ToAtt = "You grazed $defusername2.\n";
		$ToOther = "$attusername2 grazed $defusername2.\n";
	} elsif (($damage > 10) and ($damage <= 15)) {
		$ToDef = "$attusername2 hits you.\n";
		$ToAtt = "You hit $defusername2.\n";
		$ToOther = "$attusername2 hits $defusername2.\n";
	} elsif (($damage > 15) and ($damage <= 20)) {
		$ToDef = "$attusername2 hits you hard.\n";
		$ToAtt = "You hit $defusername2 hard.\n";
		$ToOther = "$attusername2 hits $defusername2 hard.\n";
	} elsif (($damage > 20) and ($damage <= 30)) {
		$ToDef = "$attusername2 hits you very hard.\n";
		$ToAtt = "You hit $defusername2 very hard.\n";
		$ToOther = "$attusername2 hits $defusername2 very hard.\n";
	} elsif (($damage > 30) and ($damage <= 50)) {
		$ToDef = "$attusername2 hits you extremly hard.\n";
		$ToAtt = "You hit $defusername2 extremly hard.\n";
		$ToOther = "$attusername2 hits $defusername2 extremly hard.\n";
	} elsif (($damage > 50) and ($damage <= 80)) {
		$ToDef = "$attusername2 smashes you with a bone crushing sound.\n";
		$ToAtt = "You smash $defusername2 with a bone crushing sound.\n";
		$ToOther = "$attusername2 smashes $defusername2 with a bone crushing sound.\n";
	} else {
		$ToDef = "$attusername2 turns you into a bloody mess.\n";
		$ToAtt = "You turn $defusername2 into a bloody mess.\n";
		$ToOther = "$attusername2 turns $defusername2 into a bloody mess.\n";
	};
	# Now send messages:
	&userTell($attcmqid, $attusername, $attpid, $attUid, $ToAtt);
	&userTell($defcmqid, $defusername, $defpid, $defUid, $ToDef);
	&informRoom($roomNum, $defUid, $ToOther, $attUid);
	return;
};

sub intCreateNpcFromProto {
	# Args: pid of summoner (0 for auto)
	#	uid of npcProto
	#	roomNumber
	# NPC roomNum is selected from: summoner's room, arg room, prototype room.
	my ($pid, $npcid, $roomNum, $args) = @_;
	my ($username, $uid, $cmqid, $gender);
	if ($pid > 0) {
		($username, $uid, $cmqid, $roomNum, $gender) = &intPid2Details($pid);
	};
	if ($main::npcLock == 1) { 
		&logPrint("Failed createNpcFromProto: lock\n");
		&clientTell($cmqid, $username, $pid, "NPC creation lock!\n")
			if $pid > 0;
		return 0; 
	};
	$main::npcLock = 1;
	# Count existing NPCs based on that proto:
	my $icnt = 0;
	foreach my $user_ref (@main::usersMainData) {
		next if (not defined $user_ref->{ npc });
		$icnt ++ if (($user_ref->{ npc } == 1) and ($user_ref->{ protoId } == $npcid));
	};
	if ($main::npcProto[ $npcid ]->{ inst_limit } <= $icnt) {
		&logPrint("Failed createNpcFromProto: inst_limit\n");
		&clientTell($cmqid, $username, $pid, "Too many NPCs from that proto!\n")
			if $pid > 0;
		$main::npcLock = 0;
		return 0;

	};
	# Copy prototype record:
	my %newNpc = %{ $main::npcProto[ $npcid ] };
	# Add missing keys
	$newNpc{ password } = 'NPC';	# Convenience.
	$newNpc{ protoId } = $npcid;
	if (defined $roomNum) { $newNpc{ room } = $roomNum; }; 	# Otherwise prototype's default.
	# Define new user id:
	my $nuid = $main::maxUserId + 1;
	# Do a redundant check of existance:
	if (defined $main::usersMainData[ $nuid ]) {
		# Somehow it exists. Increase $maxUserId and return fail.
		&logPrint("Failed createNpcFromProto: exists\n");
		&clientTell($cmqid, $username, $pid, "NPC uid conflict: $nuid\n")
			if $pid > 0;
		$main::maxUserId++;
		$main::npcLock = 0;
		return 0;
	};
	$main::usersMainData[ $nuid ] = \%newNpc;
	# Add it in the room:
	push @{ $main::roomsData[ $newNpc{ room } ]->{presentUids} }, $nuid;
	push @{ $main::roomsData[ $newNpc{ room } ]->{presentUsers} }, $newNpc{ username };
	# Also update userPass hash: 
	# This is meaningless, usernames are not unique in NPCs.
	#$main::userPass_ref->{ $newNpc{ username } } = {	password	=>	'NPC',
	#				  			uid		=>	$nuid
	#						};
	$main::maxUserId = $nuid;
	# If needed, add notifier in main loop:
	# Check for notifier code and add it to main loop:
		if (($newNpc{ npc_code } ne '') and ($newNpc{ npc_loop_secs } > 0) and ($newNpc{ npc_loop_ref } ne '')) {
			no strict "refs";
			# Load code:
			my $code = "require '" . $newNpc{ npc_code } . "'";
			eval $code;
			if ($@) {
				&logPrint("Notifier error for NPC #$nuid: $@\n");
			} else {
				my $notifier_ref = IO::Async::Timer::Periodic->new(
					interval 	=>	$newNpc{ npc_loop_secs },
					notifier_name	=>	"NPC_$nuid",
					on_tick		=>	\&{ $newNpc{ npc_loop_ref } } 
				);
				$notifier_ref->start;
				$main::loop->add($notifier_ref);
			};
		};

	$main::npcLock = 0;
	return $nuid;
};

sub npcRevival {
	# Go through @deadNpcs and either decrease counter or revive NPC.
	my @tmpDN = ();
	my $roomNum = 0;
	foreach my $idx (keys @main::deadNpcs) {
		my $uid = $main::deadNpcs[$idx];
		if (--$main::usersMainData[ $uid ]->{ revivalCounter } < 0 ) {
			# Revive NPC.
			my $protoId = $main::usersMainData[ $uid ]->{ protoId };
			# Select respawn room:
			if ($main::npcProto[ $protoId ]->{ rsp_rooms } eq '') {
				# No limitation, choose a random one:
				while ($roomNum == 0) {
					$roomNum = $main::roomsData[ rand( scalar( @main::roomsData )) ]->{ roomNum };
				};
			} else {
				# Choose from list:
				my @rsp_list = split /,/, $main::npcProto[ $protoId ]->{ rsp_rooms };
				while ($roomNum == 0) {
					$roomNum = $main::roomsData[ $rsp_list[ rand( scalar( @rsp_list )) ] ]->{ roomNum };
				};
			};
			#print "DEBUG: revival room is $roomNum\n";
			$main::usersMainData[ $uid ]->{ room } = $roomNum;
			$main::usersMainData[ $uid ]->{ hitPoints } = $main::usersMainData[ $uid ]->{ maxHp };
			my $username = $main::usersMainData[ $uid ]->{ username };
			push @{ $main::roomsData[ $roomNum ]->{presentUids} }, $uid;
			push @{ $main::roomsData[ $roomNum ]->{presentUsers} }, $username;
			&informRoom($roomNum, -1, "$username comes back from the dead!\n");
		} else {
			push @tmpDN, $uid;
		}; 
	};
	# Now replace deadNpcs with new list:
	undef @main::deadNpcs;
	@main::deadNpcs = @tmpDN;
	return;
};

sub intCmpUid2name {
	# Compare a name command argument to uid's username.
	# This is done in a different way for persons and NPCs.
	# Returns 1 on match, 0 otherwise.
	my ($uid, $name, undef) = @_;
	if ( 
		($main::usersMainData[$uid]->{ npc } == 0) and 
		(uc($main::usersMainData[$uid]->{ username }) eq uc($name)) 
	) {
		# Player. Only exact case insensitive match.
		return 1;
	};
	# For NPCs, we must only match words longer than 2 chars.
	my @words = split / /, $name;
	my @res = ();
	foreach my $word (@words) {
		if (length($word) > 2) {
			push @res, $word;
		};
	};
	$name = join(" ", @res);
	$name = quotemeta($name);
	if ($name eq '') {
		return 0;
	};
	if ( 
		($main::usersMainData[$uid]->{ npc } == 1) and 
		($main::usersMainData[$uid]->{ username } =~ m/$name/i) 
	) {
		# NPC. 
		return 1;
	};
	return 0;
};

sub createFight {
	# Will create a fight between $attUid and $defUid.
	# Assumes that all checks except fight existance
	# have been made.
	my ($attUid, $defUid, undef) = @_;
	my ($attusername, $attpid, $attcmqid, $attroomNum, $attgender) = &intUid2Details($attUid);
	my ($defusername, $defpid, $defcmqid, $defroomNum, $defgender) = &intUid2Details($defUid);
	foreach my $f (keys %main::fightData) {
		if (
			(
				($main::fightData{$f}->{ attUid } == $attUid) and
				($main::fightData{$f}->{ defUid } == $defUid)
			) or (
				($main::fightData{$f}->{ attUid } == $attUid) and
				($main::fightData{$f}->{ defUid } == $defUid)
			)
		) {
			&clientTell($attcmqid, $attusername, $attpid, "You are already fighting $defusername!\n");
			return;
		};
	};
	# Try to inform NPCs. If rc is 1, don't start the fight, NPC code 
	# takes care of the attack.
	if ($main::usersMainData[$defUid]->{ npc } == 1) {
		my $rc = &npcTell($attpid, $defUid, '__ATTACK', ''); 
		if ((defined $rc) and ($rc == 1)) {
			# All ok, return.
			return;
		};
	};
	# Create fight:
	my $fight_id = "$attUid" . "_" . "$defUid";
	$main::fightData{$fight_id}->{ attUid } = $attUid;
	$main::fightData{$fight_id}->{ defUid } = $defUid;
	$main::fightData{$fight_id}->{ timeOut } = $main::fightTimeOut;
	$main::fightData{$fight_id}->{ timeCnt } = $main::fightTimeOut;
	$main::fightData{$fight_id}->{ attDmg } = 0;
	$main::fightData{$fight_id}->{ defDmg } = 0;
	$main::fightData{$fight_id}->{ valid } = 1;
	my $attusername2 = 'Somebody';
	if ($main::usersMainData[$attUid]->{ invisible } == 0) {
		$attusername2 = $attusername;
	};
	my $defusername2 = 'Somebody';
	if ($main::usersMainData[$defUid]->{ invisible } == 0) {
		$defusername2 = $defusername;
	};
	&clientTell($attcmqid, $attusername, $attpid, "You attack $defusername2!\n");
	if ($defpid > 0) {
		&clientTell($defcmqid, $defusername, $defpid, "$attusername2 attacks you!\n");
	};
	&informRoom($attroomNum, $attUid, "$attusername2 attacks $defusername2!\n", $defUid);
	return;
};

sub randomWalk {
	# Choose a random exit and move:
	my ($roomNum, $uid, $username, undef) = @_;
	# 50% chance to stay put:
	if (rand(100) < 50) {
		#&informRoom($roomNum, $uid, "$username looks confused.\n");
		return $roomNum;
	};
	# If NPC finds itself in a room out of walk_list, teleport to a valid room.
	my $protoId = $main::usersMainData[ $uid ]->{ protoId };
	my @walk_list = split /,/, $main::npcProto[ $protoId ]->{ walk_rooms };
	if ($main::npcProto[ $protoId ]->{ walk_rooms } ne '') {
		if (scalar(grep(/^$roomNum$/, @walk_list)) < 1) {
			# Invalid room, teleport:
			my $newpos = 0;
			# Choose from list:
			while ($newpos == 0) {
				$newpos = $main::roomsData[ $walk_list[ rand( scalar( @walk_list ))] ]->{ roomNum };
			};
			#print "DEBUG: walker #$uid in invalid room (#$roomNum)! moving to #$newpos\n";
			# Inform old room!
			if ($main::usersMainData[$uid]->{ invisible } == 0) {
				&informRoom($roomNum, $uid, "$username disappears in a puf of smoke!\n");
			} else {
				&informRoom($roomNum, $uid, "There is a puf of smoke!\n");
			};
			# Move NPC:
			&movePlayer($uid, $newpos);
			# Inform new room!
			if ($main::usersMainData[$uid]->{ invisible } == 0) {
				&informRoom($newpos, $uid, "$username appears in a puf of smoke!\n");
			} else {
				&informRoom($newpos, $uid, "There is a puf of smoke!\n");
			};
			return $newpos;
		};
	};
	if (scalar(grep(/$roomNum/, @walk_list)) < 2) {
		# Nowhere to go, return:
		return $roomNum;
	};
	# Gather exits:
	my @exits = ();
	foreach my $dir (@main::directions) {
		push @exits, $dir if (($main::roomsData[$roomNum]{ "ex$dir" } > 0) and
			(scalar(grep(/$main::roomsData[$roomNum]{ "ex$dir" }/, @walk_list))) > 0);
	};
	# Choose one:
	my $ridx = scalar @exits;
	while ($ridx == scalar @exits) {
		# Using while to avoid border case.
		$ridx = rand(scalar @exits);
	};
	my $rexit = $exits[$ridx]; 
	my $longDirection = $main::dirStL{$rexit};
	my $newpos = $main::roomsData[$roomNum]{ "ex$rexit" };
	$newpos = 0 if not defined $newpos;
	if ($newpos < 1) {
		#print "DEBUG: walker could not find exit!\n";
		return 0;
	};
	# Inform old room!
	if ($main::usersMainData[$uid]->{ invisible } == 0) {
		&informRoom($roomNum, $uid, "$username goes " . lc($longDirection) . ".\n");
	};
	# Move NPC:
	&movePlayer($uid, $newpos);
	# Inform new room!
	if ($main::usersMainData[$uid]->{ invisible } == 0) {
		&informRoom($newpos, $uid, "$username just came in.\n");
	};
	return $newpos;
};

sub intTell {
	# Tell something to a player or NPC. Used by TELL and WHISPER.
	# May be called from NPC code. Returns 0 for fail, 1 for ok.
	my ($pid, $subcmd, $target, $message, $npc) = @_;
	my ($username, $uid, $cmqid, $roomNum, $gender);
	if ((defined $npc) and ($npc == 1)) {
		$uid = $pid;
		($username, $pid, $cmqid, $roomNum, $gender) = &intUid2Details($pid);
	} else {
		($username, $uid, $cmqid, $roomNum, $gender) = &intPid2Details($pid);
	};
	my %w1 = (WHISPER => 'whisper',
		  TELL => 'say');
	if ((not defined $target) or (not defined $message) or ($target eq '') or ($message eq '')) {
		if ($pid > 0) {
			&clientTell($cmqid, $username, $pid, "Please be more specific.\n");
		};
		return 0;
	};
	if (uc($username) eq uc($target)) {
		if ($pid > 0) {
			&clientTell($cmqid, $username, $pid, "YOU are $username!\n");
		};
		return 0;
	};
	my $tt_username = 'Somebody';
	if ($main::usersMainData[$uid]->{ invisible } == 0) {
		$tt_username = $username;
	};
	if (scalar(@{ $main::roomsData[ $roomNum ]->{presentUids} }) > 0) {
		foreach my $ouid (@{ $main::roomsData[ $roomNum ]->{presentUids} }) {
			# Loop through present persons and look for a username match:
			if (&intCmpUid2name($ouid, $target)) {
				# Found user.
				if ($main::usersMainData[$ouid]->{ pid } == -1) {
					if ($pid > 0) {
						&clientTell($cmqid, $username, $pid, "You talk to statues now? Hmmm...\n");
					};
					return 0;
				};
				my ($v_username, $v_pid, $v_cmqid, $v_roomNum, $v_gender) = &intUid2Details($ouid);
				if ($pid > 0) {
					&clientTell($cmqid, $username, $pid, "You $w1{$subcmd} \"$message\" to $v_username.\n");
				};
				my $tt_v_username = 'Somebody';
				if ($main::usersMainData[$ouid]->{ invisible } == 0) {
					$tt_v_username = $v_username;
				};
				if ($subcmd eq 'TELL') {
					&informRoom($roomNum, $uid, "$tt_username $w1{$subcmd}s \"$message\" to $tt_v_username.\n", $ouid);
				} else {
					&informRoom($roomNum, $uid, "$tt_username $w1{$subcmd}s something to $tt_v_username.\n", $ouid);
				};
				if ($main::usersMainData[$ouid]->{ npc } == 1) {
					# NPC
					&npcTell($pid, $ouid, '__TOLD', $message);
					return 1;
				} else {
					# User
					&clientTell($v_cmqid, $v_username, $v_pid, "$tt_username $w1{$subcmd}s to you: \"$message\".\n");
					return 1;
				};
			};
		};
		# Not found.
		if ($pid > 0) {
			&clientTell($cmqid, $username, $pid, "$target is not here.\n");
		};
		return 0;
	} else {
		if ($pid > 0) {
			&clientTell($cmqid, $username, $pid, "You are alone.\n");
		};
		return 0;
	};
};

sub intDoOther {
	# Will be called from ambience commands that have target.
	my ($pid, $subcmd, $target) = @_;
	my ($username, $uid, $cmqid, $roomNum, $gender) = &intPid2Details($pid);
	my $tt_username = 'Somebody';
	if ($main::usersMainData[$uid]->{ invisible } == 0) {
		$tt_username = $username;
	};
	if ((not defined $target) or ($target eq '')) {
		if ($pid > 0) {
			&clientTell($cmqid, $username, $pid, "Please be more specific.\n");
		};
		return 0;
	};
	if (uc($username) eq uc($target)) {
		if ($pid > 0) {
			&clientTell($cmqid, $username, $pid, "YOU are $username!\n");
		};
		return 0;
	};
	if (scalar(@{ $main::roomsData[ $roomNum ]->{presentUids} }) > 0) {
		foreach my $ouid (@{ $main::roomsData[ $roomNum ]->{presentUids} }) {
			# Loop through present persons and look for a username match:
			if (&intCmpUid2name($ouid, $target)) {
				# Found user.
				#if ($main::usersMainData[$ouid]->{ pid } == -1) {
				#	if ($pid > 0) {
				#		&clientTell($cmqid, $username, $pid, "You talk to statues now? Hmmm...\n");
				#	};
				#	return 0;
				#};
				my ($v_username, $v_pid, $v_cmqid, $v_roomNum, $v_gender) = &intUid2Details($ouid);
				if (
					($v_pid == -1) and 
					($main::usersMainData[$ouid]->{ npc } == 0)
				) {
					# Frozen user
					$v_username = "the ice statue of $v_username";
				};
				if ($pid > 0) {
					my $msg = $main::ambCmdTexts{ $subcmd }->{ TextToUser } . "$v_username.\n";
					&clientTell($cmqid, $username, $pid, $msg);
				};
				my $msg = $tt_username . $main::ambCmdTexts{ $subcmd }->{ TextToOthers } . "$v_username.\n";
				&informRoom($roomNum, $uid, $msg, $ouid);
				if ($main::usersMainData[$ouid]->{ npc } == 1) {
					# NPC
					&npcTell($pid, $ouid, '__AMB', $subcmd);
					return 1;
				} else {
					# User
					if ($v_pid > 1) {
						my $message = $tt_username . $main::ambCmdTexts{ $subcmd }->{ TextToVictim } . "\n";
						&clientTell($v_cmqid, $v_username, $v_pid, $message);
					};
					return 1;
				};
			};
		};
		# Not found.
		if ($pid > 0) {
			&clientTell($cmqid, $username, $pid, "$target is not here.\n");
		};
		return 0;
	} else {
		if ($pid > 0) {
			&clientTell($cmqid, $username, $pid, "You are alone.\n");
		};
		return 0;
	};
};

sub intMissileMove {
	my ($fromPos, $toPos, $hitFactor, $arrowFactor, $miid, $dirSTR) = @_;
	# Calculate entry direction
	my $fromLongDir = 'somewhere';
	foreach my $fromDirSTR (keys %{ $main::roomsData[$toPos] }) {
		if (($fromDirSTR =~ /^ex/) and ($main::roomsData[$toPos]{ $fromDirSTR } == $fromPos)) {
			my ($scrap, $shDir) = split /ex/, $fromDirSTR, 2;
			$fromLongDir = $main::dirStL{$shDir};
			last;
		};
	};
	&informRoom($toPos, -2, $main::itemData{ $miid }->{ shortDesc } . " shoots in from " . lc($fromLongDir) . "!\n");
	my $hitProb = 1000;
	$hitProb = $hitProb * $hitFactor / 10;	# Calculate hit probability
	if (scalar(@{ $main::roomsData[ $toPos ]->{presentUids} }) > 0) {
		SHOOTLOOP:foreach my $ouid (@{ $main::roomsData[ $toPos ]->{presentUids} }) {
			if ( 
				($main::usersMainData[ $ouid ]->{ npc } == 0) and
				($main::usersMainData[ $ouid ]->{ pid } <= 0) 
			) {
				# Frozen user
				next SHOOTLOOP;
			};

			if (rand(1000) < $hitProb) {
				# Hit!
				my ($v_username, $v_pid, $v_cmqid, $v_roomNum, $v_gender) = &intUid2Details($ouid);
				&userTell($v_cmqid, $v_username, $v_pid, $ouid, $main::itemData{ $miid }->{ shortDesc } . " hits you!\n");
				&informRoom($toPos, $ouid, $main::itemData{ $miid }->{ shortDesc } . " hits $v_username!\n");
				$main::usersMainData[ $ouid ]->{ hitPoints } -= $hitFactor * $arrowFactor;
				if ($main::usersMainData[ $ouid ]->{ hitPoints } < 10) {
					# Nobody dies from that:
					$main::usersMainData[ $ouid ]->{ hitPoints } = 10;
				};
				return 1; 	# Hit.
			};
		};
	};
	$hitFactor -= 1;			# Decrease hit factor for next iteration.
	if ($hitFactor < 1) { 
		# Out of range:
		&informRoom($toPos, -2, $main::itemData{ $miid }->{ shortDesc } . " hits the floor and breaks.\n");
		return 0;
	};
	# Check if there is an exit in the same direction:
	my $newpos = $main::roomsData[$toPos]{ $dirSTR };
	if ($newpos == 0) {
		# No exit.
		&informRoom($toPos, -2, $main::itemData{ $miid }->{ shortDesc } . " hits the stone and breaks.\n");
		return 0;
	};
	my ($scrap, $toShDir) = split /ex/, $dirSTR, 2;
	my $toLongDir = $main::dirStL{$toShDir};
	&informRoom($toPos, -2, $main::itemData{ $miid }->{ shortDesc } . " leaves " . lc($toLongDir) . ".\n");
	my $hitOk = &intMissileMove($toPos, $newpos, $hitFactor, $arrowFactor, $miid, $dirSTR);
	return $hitOk;
};

sub intAdvExp {
	my ($level, undef) = @_;
	# Calculate experience for level:
	if ($level == 0) {
		return 0;
	} else {
		my $res = &intAdvExp($level-1) + ($level**3)*50;
		return $res; 
	};
};

sub cmpItems {
	# Compare two item records and return true if they are identical.
	# Some item fields are not compared.
	my ($iid1, $iid2, undef) = @_;
	my @exceptions = ('id', 'created_by', 'amount');
	foreach my $attr (keys %{ $iid1 }) {
		if (scalar(grep(/$attr/, @exceptions)) > 0) {
			# Exception
			next;
		};
		if (
			(not defined $iid1->{ $attr }) and 
			(not defined $iid2->{ $attr })  
		) {
			# Exception
			next;
		};
		if (not defined $iid1->{ $attr }) { $iid1->{ $attr } = ''; }; 
		if (not defined $iid2->{ $attr }) { $iid2->{ $attr } = ''; }; 

		if ($iid1->{ $attr } ne $iid2->{ $attr }) {
			return 0;
		};
	};
	return 1;
};

sub intInvSelectNum {
	my ($uid, $attrib, $value, undef) = @_;
	# Find items for this uid that have $atrrib set to $value
	# and return iids as array. For example, 'wielded' and 1 
	# returns all wielded items. Works for numeric values only.
	my @res = ();
	foreach my $iid (keys %{ $main::usersMainData[ $uid ]->{ inventory } }) {
		if ($main::usersMainData[ $uid ]->{ inventory }->{$iid}->{ $attrib } == $value) {
			push @res, $iid;
		};
	};
	return @res;
};

sub intConsolidateInv {
	# Consolidate the user's or room's inventory referenced by $inv_ref
	my ($inv_ref, undef) = @_;
	my %identItems = ();	# Store pairs of identical items for consolidation.
	# Detect and store identical items:

	foreach my $iid (keys %{ $inv_ref }) {
		foreach my $liid (keys %{ $inv_ref }) {
			if (
				(&cmpItems($inv_ref->{ $iid }, $inv_ref->{ $liid })) and
				($iid != $liid)
			) {
				$identItems{$iid} = $liid;
			};
		};
	};


	# Consolidate identical items:
	foreach my $iid (keys %identItems) {
		my $liid = $identItems{$iid};
		if (
			(defined $inv_ref->{ $iid }) and
			(defined $inv_ref->{ $liid }) and
			# Don't try to consolidate coin items.
			($inv_ref->{ $iid }->{ gen_item_type } != 13) and
			# Don't consolidate worn or wielded items.
			($inv_ref->{ $iid }->{ worn } == 0) and
			($inv_ref->{ $iid }->{ wielded } == 0) and
			($inv_ref->{ $liid }->{ worn } == 0) and
			($inv_ref->{ $liid }->{ wielded } == 0)
		) {
			# Increase the amount of first item and destroy the second one.
			$inv_ref->{ $iid }->{ amount } += $inv_ref->{ $liid }->{ amount };
			&intDestroyItem($liid);
		};
	};
};

sub defAlignement {
	my ($numAli, undef) = @_;
	if ($numAli < -250) {
		return "Totally Evil";
	} elsif (($numAli >= -250) and ($numAli < -150)) {
		return "Evil";
	} elsif (($numAli >= -150) and ($numAli < -50)) {
		return "Slightly Evil";
	} elsif (($numAli >= -50) and ($numAli < 50)) {
		return "Neutral";
	} elsif (($numAli >= 50) and ($numAli < 150)) {
		return "Good";
	} elsif (($numAli >= 150) and ($numAli < 250)) {
		return "Paladine";
	} elsif ($numAli >= 250) {
		return "Saint";
	};
};

END { } # module clean-up code here (global destructor)

1; # don't forget to return a true value from the file
