#############################################################################
# MUDA: Resurrection							    #
# Ms_ParseCommand.pm	: Main command parsing code.			    #
# Artist:		: Theodore J. Soldatos				    #
#############################################################################
# This file is part of MUDA:Resurrection.			   	    #
#									    #
# MUDA:Resurrection is free software: you can redistribute it and/or modify #
# it under the terms of the GNU General Public License as published by	    #
# the Free Software Foundation, either version 3 of the License, or	    #
# (at your option) any later version.					    #
#									    #
# MUDA:Resurrection is distributed in the hope that it will be useful,	    #
# but WITHOUT ANY WARRANTY; without even the implied warranty of	    #
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the		    #
# GNU General Public License for more details.				    #
#									    #
# You should have received a copy of the GNU General Public License	    #
# along with MUDA:Resurrection.  If not, see <http://www.gnu.org/licenses/>.#
#									    #
# Copyright 2012 Theodore J. Soldatos					    #
#############################################################################

package Ms_Libs;

use strict;
no strict "refs";
use warnings;

use IPC::SysV qw(IPC_PRIVATE IPC_RMID IPC_CREAT S_IRWXU IPC_NOWAIT);
use Text::Autoformat;

BEGIN {
use Exporter ();
our ($VERSION, @ISA, @EXPORT, @EXPORT_OK, %EXPORT_TAGS);
# set the version for version checking
$VERSION = 1.00;
@ISA = qw(Exporter);
@EXPORT = qw(&parseCommand);
%EXPORT_TAGS = ( ); # eg: TAG => [ qw!name1 name2! ],
#@EXPORT_OK = qw();
}
our @EXPORT_OK;

sub parseCommand {
	# This routine will decide which sub to call and with what arguments for most 
	# of the user commands. The reason is that a command may be made available not
	# only through the main game's commands list (which is stored in the DB and 
	# loaded from &loadCommands() into $main::cmdhash_ref), but through other 
	# sources, like inventory item commands, room item commands or NPCs. 
	# For now (Oct 7 2012), only the main list is available.
	# Item commands also available (Oct 11 2012)
	my ($clientPID, $clientCMD, $clientArgs) = @_;
	if (not defined ($main::pid2username_ref->{$clientPID})) {
		&logPrint("Unknown pid $clientPID in parseCommand sub.\n");
		# Discard command to avoid server crash.
		# Try to kick out zombie client:
		kill('INT', $clientPID);
		return;
	};

	my ($username, $uid, $cmqid, $roomNum, $gender) = &intPid2Details($clientPID);
	my $cmd2exe = '';
	# Go through commands list(s) matching the full string. Precedence will be defined 
	# by code order.

	my %items_cmds;
	my %scores;
	# Inventory available commands (stored in items in user's inventory):
	# Room available commands (stored in items found in the room):
	my @roomItems = keys %{ $main::roomsData[ $roomNum ]->{ presentItems } };
	my @invItems = keys %{ $main::usersMainData[ $uid ]->{ inventory } };
	my @allItems = (@invItems, @roomItems);
	ALLIT:	foreach my $iid (@allItems) {
		my $item_ref = $main::itemData{ $iid };
		if (not defined $item_ref->{ executable }) { 
			$item_ref->{ executable } = '';
		};
		if (
			($item_ref->{ executable } ne '') and
			($item_ref->{ validCode } == 1)
		) {
			# Check this item:
			#delete $INC{ $item_ref->{ executable } };
			#my $code = "require \'" . $item_ref->{ executable } . "\';";
			#eval $code;
			#if ($@) {
			#	warn $@;
			#	next ALLIT;
			#};
			my $iCmdList = &{ $item_ref->{ code_ref } }($clientPID, '__LIST');
			my @iCmdList_arr = split / /, &{ $item_ref->{ code_ref } }($clientPID, '__LIST');
			$items_cmds{ $iid } = {
				id 		=> $iid,
				cmdlist		=> $iCmdList,
				shortDesc 	=> $item_ref->{ shortDesc },
			};
			$scores{ $iid } = 0;
			#if ($iCmdList =~ m/^$clientCMD/i) {
			if (grep(m/^$clientCMD/i, @iCmdList_arr)) {
				# Item has requested command. 
				$scores{ $iid } += 100;
			};
			# A simple way to distinguish between two items providing the same
			# command: match command arguments to item's description. For most
			# cases, this should be enough (TM). 
			my @argwords = split / /, $clientArgs;
			foreach my $word(@argwords) {
				$word = quotemeta($word);
				if ($items_cmds{ $iid }->{ shortDesc } =~ m/$word/i) {
					$scores{ $iid }++;
				};
				# Also check numeric item (#xxx) case:
				my ($scrap, $spiid) = split /#/, $word, 2;
				if (not defined $spiid) { $spiid = -1; };
				if ($iid == $spiid) {
					# This should be in the top, maximum match:
					$scores{ $iid } += 1000;
				};
			};
			#&logPrint("DEBUG: Item #$iid (" . $items_cmds{ $iid }->{ shortDesc } . ") scores " . $scores{ $iid } . "\n");
		};
	};

	# Now go through %items_cmds and use the item with maximum score.
	# ... but first remove items with no command match:
	if (scalar(keys %scores) > 0) {
		foreach my $iid (keys %scores) {
			if ($scores{$iid} < 100) {
				delete $scores{$iid};
			};
		};
	};

	my @sorted_items = sort { $scores{$a} <=> $scores{$b} } keys %scores;
	# DEBUG code
	#foreach my $itemid (@sorted_items) {
	#	print $items_cmds{$itemid}->{ shortDesc } . " has " . $scores{$itemid} . "\n";
	#};
	# END DEBUG
	#SELECT: for (my $i = 1; $i < 2; $i++) {
	SELECT: while (my $top_iid = pop @sorted_items) {
		# In this point, we completely rely on item code to behave
		# properly and return the right rc. We call the code of all
		# items with score >= 100, from highest to lowest.
		# As soon as an item says that did something, we return.
		# If no item does anything, continue to normal commands.

		#if (scalar(keys %scores) > 0) {
		#my $top_iid = pop @sorted_items;
		#print "DEBUG: Working on: " . $items_cmds{$top_iid}->{ shortDesc } . " has " . $scores{$top_iid} . "\n";
		#my $sec_iid = pop @sorted_items;
		#if (defined $sec_iid) {
		#	if ($scores{$top_iid} == $scores{$sec_iid}) {
		#		# More than one, ambiguous, ignore item commands.
		#		#&clientTell($cmqid, $username, $clientPID, "Please be more specific.\n");
		#		#return;
		#		last SELECT;
		#	};
		#};
		# Execute item command and return:
		#print "DEBUG: Executing...\n";
		my $item_ref = $main::itemData{ $top_iid };
		my $item_rc = &{ $item_ref->{ code_ref } }($clientPID, uc($clientCMD), $top_iid, $clientArgs);
		if (not defined $item_rc) {
			&logPrint("Command failed for item #$top_iid (" . $item_ref->{ code_ref } . ")\n");
		};
		if ($item_rc) {
			# Item worked, don't check other commands.
			return;
		} else {
			# Item returned 0, continue to other items and standard commands.
		};
		#};
	};

	# Globally available commands (DB stored):
	foreach my $possCmd (keys %$main::cmdhash_ref) {
		if (
			($clientCMD eq $possCmd) and 
			($main::cmdhash_ref->{ $possCmd}->{ reqLevel } <= $main::usersMainData[ $uid ]->{ level })
		) {
			# Perfect match. These cmds are unique, so no reason to search more.
			$cmd2exe = $possCmd;
			last;
		};
	};
	if ($cmd2exe ne '') {
		# Execute and return:
		my $subName = $main::cmdhash_ref->{ $cmd2exe }->{ proc };
		&{ $subName }($clientPID, $clientArgs);
		return;
	};
	# Else continue:
	my @cmd2exe = ();
	# Now search for substrings:
	foreach my $possCmd (keys %$main::cmdhash_ref) {
		if (
			($possCmd =~ m/^$clientCMD/i) and 
			($main::cmdhash_ref->{ $possCmd}->{ reqLevel } <= $main::usersMainData[ $uid ]->{ level })
		) {
			push @cmd2exe, $possCmd;
		};
	};
	if (scalar(@cmd2exe) > 1) {
		# Too many.
		&clientTell($cmqid, $username, $clientPID, "Ambiguous command $clientCMD\n");
		return;
	} elsif (scalar(@cmd2exe < 1)) {
		# Nothing found.
		&cmdUnknown($clientPID, $clientCMD);
		return;
	} else {
		# Execute and return:
		$cmd2exe = pop @cmd2exe;
		my $subName = $main::cmdhash_ref->{ $cmd2exe }->{ proc };
		&{ $subName }($clientPID, $clientArgs);
		return;
	};
	# Should not get here, just to be sure:
	&clientTell($cmqid, $username, $clientPID, "Fall off command $clientCMD! Please contact a God!\n");
	return;
};
END { } # module clean-up code here (global destructor)

1; # don't forget to return a true value from the file
