#############################################################################
# MUDA: Resurrection							    #
# Ms_Cons_Subs.pm	: System routines and "console" commands.	    #
# Artist:		: Theodore J. Soldatos				    #
#############################################################################
# This file is part of MUDA:Resurrection.			   	    #
#									    #
# MUDA:Resurrection is free software: you can redistribute it and/or modify #
# it under the terms of the GNU General Public License as published by	    #
# the Free Software Foundation, either version 3 of the License, or	    #
# (at your option) any later version.					    #
#									    #
# MUDA:Resurrection is distributed in the hope that it will be useful,	    #
# but WITHOUT ANY WARRANTY; without even the implied warranty of	    #
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the		    #
# GNU General Public License for more details.				    #
#									    #
# You should have received a copy of the GNU General Public License	    #
# along with MUDA:Resurrection.  If not, see <http://www.gnu.org/licenses/>.#
#									    #
# Copyright 2012 Theodore J. Soldatos					    #
#############################################################################


package Ms_Libs;

use strict;
use warnings;

use DateTime;
use DateTime::Format::SQLite;

BEGIN {
use Exporter ();
our ($VERSION, @ISA, @EXPORT, @EXPORT_OK, %EXPORT_TAGS);
$VERSION = 1.00;
@ISA = qw(Exporter);
@EXPORT = qw(&printRoom &printLong &loadUserData &saveSingleUser &saveAllUsers &saveLoggedInUsers &loadRooms &loadItemData &loadItemProto &loadItemTypes &loadCommands &saveAllItems &reloadItemLoopCode &mainFightTick &loadNpcProto &saveNewUser &intEOFightCheck &loadQuests &saveAllQuests &saveAllUsersQuests &reloadItemCode &reloadNpcCode &intMetabolism &checkRCMDintegrity);
%EXPORT_TAGS = ( ); # eg: TAG => [ qw!name1 name2! ],
}
our @EXPORT_OK;

sub printRoom {
	# Print current room:
	my ($pos, undef) = @_;

	print "Room #$pos\n\n";
	printLong("$main::roomsLongDescr[$pos]");

	print "Exits:\n";

	print "East:\t$main::roomsShortDescr[ $main::roomsData[$pos]{exE} ]\n" if ($main::roomsData[$pos]{exE} != 0);
	print "West:\t$main::roomsShortDescr[ $main::roomsData[$pos]{exW} ]\n" if ($main::roomsData[$pos]{exW} != 0);
	print "North:\t$main::roomsShortDescr[ $main::roomsData[$pos]{exN} ]\n" if ($main::roomsData[$pos]{exN} != 0);
	print "South:\t$main::roomsShortDescr[ $main::roomsData[$pos]{exS} ]\n" if ($main::roomsData[$pos]{exS} != 0);
	print "Up:\t$main::roomsShortDescr[ $main::roomsData[$pos]{exU} ]\n" if ($main::roomsData[$pos]{exU} != 0);
	print "Down:\t$main::roomsShortDescr[ $main::roomsData[$pos]{exD} ]\n" if ($main::roomsData[$pos]{exD} != 0);
	print "NE:\t$main::roomsShortDescr[ $main::roomsData[$pos]{exNE} ]\n" if ($main::roomsData[$pos]{exNE} != 0);
	print "NW:\t$main::roomsShortDescr[ $main::roomsData[$pos]{exNW} ]\n" if ($main::roomsData[$pos]{exNW} != 0);
	print "SE:\t$main::roomsShortDescr[ $main::roomsData[$pos]{exSE} ]\n" if ($main::roomsData[$pos]{exSE} != 0);
	print "SW:\t$main::roomsShortDescr[ $main::roomsData[$pos]{exSW} ]\n" if ($main::roomsData[$pos]{exSW} != 0);
	return $pos;
};

sub printLong {
	my ($line, undef) = @_;
	while ($line ne '')  {
		if (length($line) > $main::termWidth) {
			print substr($line, 0, $main::termWidth), "\n";
			$line = substr($line, $main::termWidth);
		} else {
			print $line, "\n";
			$line = '';
		};
	};
};

sub loadRooms {
	# Load rooms from SQL DB.
	use DBD::SQLite;
	my $dbh = DBI->connect("dbi:SQLite:dbname=$main::dbfile","",""); 
	my ($id, $shortDescr, $longDescr, $exE, $exW, $exN, $exS, $exU, $exD, $exNE, $exNW, $exSE, $exSW);
	my $query = "SELECT id, shortDescr, longDescr, exE, exW, exN, exS, exU, exD, exNE, exNW, exSE, exSW FROM roomData";
	my $sth  = $dbh->prepare($query) 
		or die "prepare: " . $dbh->errstr();;
	$sth->execute() 
		or die "execute: " . $dbh->errstr();;
	$sth->bind_columns(\$id, \$shortDescr, \$longDescr, \$exE, \$exW, \$exN, \$exS, \$exU, \$exD, \$exNE, \$exNW, \$exSE, \$exSW)
        	or die "bind_columns: " . $dbh->errstr();
	my $roomcnt = 0;
	while ($sth->fetch) {
		$main::roomsLongDescr[$id] = $longDescr;
		$main::roomsShortDescr[$id] = $shortDescr;
		my @tmpPresentUids = [];
		my @tmpPresentUsers = [];
		my $tmpPresentItems_ref = {};
		my %tmpRoom = (
			exE 		=>	$exE,
			exW 		=>	$exW,
			exN 		=>	$exN,
			exS 		=>	$exS,
			exU 		=>	$exU,
			exD 		=>	$exD,
			exNE 		=>	$exNE,
			exNW 		=>	$exNW,
			exSE 		=>	$exSE,
			exSW 		=>	$exSW,
			roomNum		=>	$id,
			presentUids	=>	@tmpPresentUids,	
			presentUsers	=>	@tmpPresentUsers,
			presentItems	=>	$tmpPresentItems_ref,
		);
		$main::roomsData[$id] = \%tmpRoom;
		$roomcnt++;
	};
	$sth->finish();
	&logPrint("Loaded $roomcnt rooms.\n");
	$main::stats_ref->{ roomCount } = $roomcnt;
	$dbh->disconnect;
};

sub loadUserData {
	#my ($fn, $rest) = @_;
	#use DBD::CSV;
	use DBD::SQLite;
	my $dbh = DBI->connect("dbi:SQLite:dbname=$main::dbfile","",""); 
	my ($uid, $username, $password, $desc, $level, $coins, $xp, $room, $godMode, $email, $gender, $paralyzed, $invisible, $invincible, $blind, $hitPoints, $mana, $maxHp, $maxMana, $lastCon, $lastDiscon, $lastSave, $npc, $npc_code_ref, $npc_loop_ref, $npc_loop_secs, $npc_code, $protoId, $revivalTime, $poisoned, $intoxicated, $fuel, $memory, $alignement);
	my $query = "SELECT UID, username, password, description, level, coins, XP, room, godmode, email, gender, paralyzed, invisible, invincible, blind, hitPoints, mana, maxHp, maxMana, lastCon, lastDiscon, lastSave, npc, npc_code_ref, npc_loop_ref, npc_loop_secs, npc_code, protoId, revivalTime, poisoned, intoxicated, fuel, memory, alignement FROM mainUserData";
	my $sth  = $dbh->prepare($query) 
		or die "prepare: " . $dbh->errstr();;
	$sth->execute() 
		or die "execute: " . $dbh->errstr();;
	$sth->bind_columns(\$uid, \$username, \$password, \$desc, \$level, \$coins, \$xp, \$room, \$godMode, \$email, \$gender, \$paralyzed, \$invisible, \$invincible, \$blind, \$hitPoints, \$mana, \$maxHp, \$maxMana, \$lastCon, \$lastDiscon, \$lastSave, \$npc, \$npc_code_ref, \$npc_loop_ref, \$npc_loop_secs, \$npc_code, \$protoId, \$revivalTime, \$poisoned, \$intoxicated, \$fuel, \$memory, \$alignement)
        	or die "bind_columns: " . $dbh->errstr();
	my $usercntr = 0;
	my $npccntr = 0;
	my $codenpccntr = 0;
	my $loopnpccntr = 0;
	while ($sth->fetch) {
		#print("Found result row:\n");
		#print "\tuid=$uid\n";
		#print "\tusername=$username\n";
		#print "\tpassword=$password\n";
		#print "\tdesc=$desc\n";
		#print "\tlevel=$level\n";
		#print "\tcoins=$coins\n";
		#print "\txp=$xp\n";
		#print "\troom=$room\n";
		my $tmpPresentItems_ref = {};
		# Store in array of hashes:
		$main::usersMainData[$uid] = { 	username 	=> $username,
						password 	=> $password,
						desc	 	=> $desc,
						level	 	=> $level,
						coins	 	=> $coins,
						xp	 	=> $xp,
						room	 	=> $room,
						pid	 	=> -1,		# Not logged in.
						qid	 	=> -1,		# Not logged in.
						timeout	 	=> 0,		# Inactivity counter.
						validCode	=> 0,		# If coded NPC, becomes 1 when code is 
										# successfully parsed.
						godMode	 	=> $godMode,
						email	 	=> $email,
						gender	 	=> $gender,
						paralyzed 	=> $paralyzed,
						invisible 	=> $invisible,
						invincible 	=> $invincible,		# No fights possible at all.
						blind	 	=> $blind,
						hitPoints 	=> $hitPoints,
						mana	 	=> $mana,
						maxHp 		=> $maxHp,
						maxMana	 	=> $maxMana,
						lastCon	 	=> $lastCon,
						lastDiscon	=> $lastDiscon,
						lastSave	=> $lastSave,		
						inventory	=> $tmpPresentItems_ref,# User's inventory
						npc		=> $npc,		# If true, is NPC
						npc_code_ref	=> $npc_code_ref,	# NPC main sub ref
						npc_loop_ref	=> $npc_loop_ref,	# NPC loop sub ref
						npc_loop_secs	=> $npc_loop_secs,	# Period of NPC loop
						npc_code	=> $npc_code,		# NPC code filename
						protoId		=> $protoId,		# NPC prototype ID 
						revivalTime	=> $revivalTime,	# NPC revival time
						quests		=> {},			# Completed quests
						poisoned	=> $poisoned,		# 
						intoxicated	=> $intoxicated,	# 
						fuel		=> $fuel,		# 
						memory		=> $memory,		# Memory field for NPCs. 
						alignement	=> $alignement,		#  
					};
		# Thaw memory data:
		if ($memory ne '') {
			# Use stored memory data structure
			my @tmp = FreezeThaw::thaw $memory;
			$main::usersMainData[$uid]->{ memory } = \%{ $tmp[0] };
		} else {
			# Initialize memory data structure
			my %tmphash = ();
			$main::usersMainData[$uid]->{ memory } = \%tmphash;
		}; 

		# Make NPC more NPC:
		if ($main::usersMainData[$uid]->{ npc }) {
			$main::usersMainData[$uid]->{ pid } = -2;
			$main::usersMainData[$uid]->{ qid } = -2;
			$main::usersMainData[$uid]->{ revivalCounter } = $revivalTime;	# In case of death...
		};
		# Update dead NPCs array:
		if (
			($main::usersMainData[$uid]->{ npc } == 1) and 
			($main::usersMainData[$uid]->{ room } < 0)
		) {
			push @main::deadNpcs, $uid;
		};
		push @{ $main::roomsData[ $room ]->{presentUids} }, $uid;
		push @{ $main::roomsData[ $room ]->{presentUsers} }, $username;
		# Also update userPass hash: 
		# This should NOT be used to work on NPCs. NPC's usernames are NOT 
		# unique, so this hash contains wrong data. Must be used only 
		# in login or in conjuction with loggeInUsers.
		$main::userPass_ref->{ $username } = {	password	=>	$password,
					  		uid		=>	$uid
							};
		if ($uid > $main::maxUserId) { $main::maxUserId = $uid };
		
		# Load user's solved quests:
		my ($uqid, $qid);
		my $lquery = "SELECT id, qid FROM userQuests WHERE uid = $uid";
		my $lsth  = $dbh->prepare($lquery) 
			or die "prepare: " . $dbh->errstr();;
		$lsth->execute() 
			or die "execute: " . $dbh->errstr();;
		$lsth->bind_columns(\$uqid, \$qid)
       		 	or die "bind_columns: " . $dbh->errstr();
		while ($lsth->fetch) {
			$main::usersMainData[$uid]->{ quests }->{ $uqid } = $qid;
		};

		# Check for notifier code and add it to main loop:
		if (($npc_code ne '') and ($npc_loop_secs > 0) and ($npc_loop_ref ne '')) {
			no strict "refs";
			# Load code:
			my $code = "require '" . $npc_code . "'";
			eval $code;
			if ($@) {
				&logPrint("Notifier error for NPC #$uid: $@\n");
			} else {
				my $notifier_ref = IO::Async::Timer::Periodic->new(
					interval 	=>	$npc_loop_secs,
					notifier_name	=>	"NPC_$uid",
					on_tick		=>	\&{ $npc_loop_ref } 
				);
				$notifier_ref->start;
				$main::loop->add($notifier_ref);
			};
		};

		$usercntr++;
		$npccntr++ if $npc; 
		$codenpccntr++ if $npc_code ne ''; 
		$loopnpccntr++ if $npc_loop_ref ne ''; 
	};
	$sth->finish();
	&logPrint( "Loaded $usercntr users ($npccntr NPCs, $codenpccntr coded, $loopnpccntr loop).\n");
	$main::stats_ref->{ userCount } = $usercntr;
	$main::stats_ref->{ npcCount } = $npccntr;
	$main::stats_ref->{ codeNpcCount } = $codenpccntr;
	$main::stats_ref->{ loopnpcCount } = $loopnpccntr;
	$dbh->disconnect;
};

sub loadItemData {
	use DBD::SQLite;
	my $dbh = DBI->connect("dbi:SQLite:dbname=$main::dbfile","",""); 
	my ($id, $shortDesc, $longDesc, $gen_item_type, $wearable, $wieldable, $movable, $invisible, $weight, $volume, $containVolume, $dropable, $created_by, $edible, $drinkable, $key, $missile, $missile_type, $missile_launcher, $missile_type_needed, $executable, $code_ref, $loop_ref, $loop_secs, $armourClass, $weaponClass, $amount, $roomID, $userID, $itemID, $worn, $wielded, $keepOnDeath, $questCode, $poison, $healing, $intox);
	my $query = 'SELECT id, shortDesc, longDesc, gen_item_type, wearable, wieldable, movable, invisible, weight, volume, containVolume, dropable, created_by, edible, drinkable, key, missile, missile_type, missile_launcher, missile_type_needed, executable, code_ref, loop_ref, loop_secs, armourClass, weaponClass, amount, roomID, userID, itemID, worn, wielded, keepOnDeath, questCode, poison, healing, intox FROM itemData';
	my $sth  = $dbh->prepare($query) 
		or die "prepare: " . $dbh->errstr();;
	$sth->execute() 
		or die "execute: " . $dbh->errstr();;
	$sth->bind_columns(\$id, \$shortDesc, \$longDesc, \$gen_item_type, \$wearable, \$wieldable, \$movable, \$invisible, \$weight, \$volume, \$containVolume, \$dropable, \$created_by, \$edible, \$drinkable, \$key, \$missile, \$missile_type, \$missile_launcher, \$missile_type_needed, \$executable, \$code_ref, \$loop_ref, \$loop_secs, \$armourClass, \$weaponClass, \$amount, \$roomID, \$userID, \$itemID, \$worn, \$wielded, \$keepOnDeath, \$questCode, \$poison, \$healing, \$intox);
	my $itemcntr = 0;
	my $loopcntr = 0;
	my $codecntr = 0;
	while ($sth->fetch) {
		my $rec = {
	 		id			=>	$id, 
			shortDesc		=>	$shortDesc, 
			longDesc		=>	$longDesc, 
			gen_item_type		=>	$gen_item_type,
			wearable		=>	$wearable,
			wieldable		=>	$wieldable,
			movable			=>	$movable,
			invisible		=>	$invisible,
			weight			=>	$weight,
			volume			=>	$volume,
			containVolume		=>	$containVolume,
			dropable		=>	$dropable,
			created_by		=>	$created_by,
			edible			=>	$edible,
			drinkable		=>	$drinkable,
			key			=>	$key,
			missile			=>	$missile,
			missile_type		=>	$missile_type,
			missile_launcher	=>	$missile_launcher,
			missile_type_needed	=>	$missile_type_needed,
			executable		=>	$executable,
			code_ref		=>	$code_ref,
			loop_ref		=>	$loop_ref,
			loop_secs		=>	$loop_secs,
			armourClass		=>	$armourClass, 
			weaponClass		=>	$weaponClass, 
			amount			=>	$amount, 
			roomID			=>	$roomID, 
			userID			=>	$userID, 
			itemID			=>	$itemID, 
			worn			=>	$worn, 
			wielded			=>	$wielded, 
			keepOnDeath		=>	$keepOnDeath, 
			questCode		=>	$questCode, 
			validCode		=>	0,	# Not saved. True when code exists
								# and evaluates without errors.
			poison			=>	$poison, 
			healing			=>	$healing, 
			intox			=>	$intox, 
		};
		# Main item data structure, used for mass saves:
		$main::itemData{ $id } = $rec;
		# Item in user's inventory go to user data structure:
		if ($userID > 0) {
			# User item. No need to check itemID, contained items have both 
			# fields set.
			$main::usersMainData[ $userID ]->{ inventory }->{ $id } = $rec; 
		} elsif ($roomID > 0) {
			# Laying around...
			$main::roomsData[ $roomID ]->{ presentItems }->{ $id } = $rec;
		};
		if ($id > $main::maxItemId) { $main::maxItemId = $id };
		$itemcntr++;
		# Check for code (just for count).
		if ($rec->{ code_ref } ne '') {
			$codecntr++;
		};
		# Check for notifier code and add it to main loop:
		if (($rec->{ loop_ref } ne '') and ($rec->{ loop_secs } > 0)) {
			no strict "refs";
			# Load code:
			my $code = "require '" . $rec->{ executable } . "'";
			eval $code;
			if ($@) {
				&logPrint( "Notifier error for item #$id: $@\n");
			} else {
				my $notifier_ref = IO::Async::Timer::Periodic->new(
					interval 	=>	$rec->{ loop_secs },
					notifier_name	=>	"ITEM_$id",
					on_tick		=>	\&{ $rec->{ loop_ref } }
				);
				$notifier_ref->start;
				$main::loop->add($notifier_ref);
				$loopcntr++;
			};
		};
	};
	$sth->finish();
	&logPrint( "Loaded $itemcntr items ($codecntr programmed, $loopcntr with loop code).\n");
	$main::stats_ref->{ itemsCount } = $itemcntr;
	$main::stats_ref->{ codedItemsCount } = $codecntr;
	$main::stats_ref->{ loopItemsCount } = $loopcntr;
	$dbh->disconnect;
};

sub loadItemProto {
	use DBD::SQLite;
	my $dbh = DBI->connect("dbi:SQLite:dbname=$main::dbfile","",""); 
	my ($id, $shortDesc, $longDesc, $gen_item_type, $wearable, $wieldable, $movable, $invisible, $weight, $volume, $containVolume, $dropable, $created_by, $edible, $drinkable, $key, $missile, $missile_type, $missile_launcher, $missile_type_needed, $executable, $code_ref, $loop_ref, $loop_secs, $armourClass, $weaponClass, $amount, $keepOnDeath, $questCode, $poison, $healing, $intox);
	my $query = 'SELECT id, shortDesc, longDesc, gen_item_type, wearable, wieldable, movable, invisible, weight, volume, containVolume, dropable, created_by, edible, drinkable, key, missile, missile_type, missile_launcher, missile_type_needed, executable, code_ref, loop_ref, loop_secs, armourClass, weaponClass, amount, keepOnDeath, questCode, poison, healing, intox FROM itemProto';
	my $sth  = $dbh->prepare($query) 
		or die "prepare: " . $dbh->errstr();;
	$sth->execute() 
		or die "execute: " . $dbh->errstr();;
	$sth->bind_columns(\$id, \$shortDesc, \$longDesc, \$gen_item_type, \$wearable, \$wieldable, \$movable, \$invisible, \$weight, \$volume, \$containVolume, \$dropable, \$created_by, \$edible, \$drinkable, \$key, \$missile, \$missile_type, \$missile_launcher, \$missile_type_needed, \$executable, \$code_ref, \$loop_ref, \$loop_secs, \$armourClass, \$weaponClass, \$amount, \$keepOnDeath, \$questCode, \$poison, \$healing, \$intox);
	my $itemcntr = 0;
	my $codecntr = 0;
	my $loopcntr = 0;
	while ($sth->fetch) {
		my $rec = {
	 		id			=>	$id, 
			shortDesc		=>	$shortDesc, 
			longDesc		=>	$longDesc, 
			gen_item_type		=>	$gen_item_type,
			wearable		=>	$wearable,
			wieldable		=>	$wieldable,
			movable			=>	$movable,
			invisible		=>	$invisible,
			weight			=>	$weight,
			volume			=>	$volume,
			containVolume		=>	$containVolume,
			dropable		=>	$dropable,
			created_by		=>	$created_by,
			edible			=>	$edible,
			drinkable		=>	$drinkable,
			key			=>	$key,
			missile			=>	$missile,
			missile_type		=>	$missile_type,
			missile_launcher	=>	$missile_launcher,
			missile_type_needed	=>	$missile_type_needed,
			executable		=>	$executable,
			code_ref		=>	$code_ref,
			loop_ref		=>	$loop_ref,
			loop_secs		=>	$loop_secs,
			armourClass		=>	$armourClass, 
			weaponClass		=>	$weaponClass, 
			amount			=>	$amount, 
			keepOnDeath		=>	$keepOnDeath, 
			questCode		=>	$questCode, 
			poison			=>	$poison, 
			healing			=>	$healing, 
			intox			=>	$intox, 
		};
		# Main item prototype data structure
		$main::itemProto{ $id } = $rec;
		$itemcntr++;
		if (not defined $rec->{ code_ref }) { $rec->{ code_ref } = ''; };
		if (not defined $rec->{ loop_ref }) { $rec->{ loop_ref } = ''; };
		if ($rec->{ code_ref } ne '') {
			$codecntr++;
		};
		if ($rec->{ loop_ref } ne '') {
			$loopcntr++;
		};
	};
	$sth->finish();
	&logPrint( "Loaded $itemcntr item prototypes ($codecntr programmed, $loopcntr with loop code).\n");
	$main::stats_ref->{ itemProtoCount } = $itemcntr;
	$main::stats_ref->{ codedItemProtoCount } = $codecntr;
	$main::stats_ref->{ loopItemProtoCount } = $loopcntr;
	$dbh->disconnect;
};

sub loadItemTypes {
	use DBD::SQLite;
	my $dbh = DBI->connect("dbi:SQLite:dbname=$main::dbfile","",""); 
	my ($itid , $descr, $wwLimit);
	my $query = 'SELECT itid, descr, wwLimit FROM itemTypes';
	my $sth  = $dbh->prepare($query) 
		or die "prepare: " . $dbh->errstr();;
	$sth->execute() 
		or die "execute: " . $dbh->errstr();;
	$sth->bind_columns(\$itid , \$descr, \$wwLimit);
	my $typecnt = 0;
	while ($sth->fetch) {
		my $rec = {
			itid		=>	$itid,
			descr		=>	$descr,
			wwLimit		=>	$wwLimit,
		};
		$main::itemTypes{ $itid } = $rec;
		$typecnt++;
	};
	$sth->finish();
	&logPrint("Loaded $typecnt item types.\n");
	$main::stats_ref->{ itemTypes } = $typecnt;
	$dbh->disconnect;
};

sub loadCommands {
	use DBD::SQLite;
	my $dbh = DBI->connect("dbi:SQLite:dbname=$main::dbfile","",""); 
	my ($id, $cmd, $proc, $xtra, $reqLevel);
	my $query = 'SELECT id, cmd, proc, xtra, reqLevel FROM commandList';
	my $sth  = $dbh->prepare($query) 
		or die "prepare: " . $dbh->errstr();;
	$sth->execute() 
		or die "execute: " . $dbh->errstr();;
	$sth->bind_columns(\$id, \$cmd, \$proc, \$xtra, \$reqLevel);
	my $cmdcnt = 0;
	while ($sth->fetch) {
		$main::cmdhash_ref->{ $cmd } = {	
						id		=> $id,
						cmd		=> $cmd,
						proc		=> $proc,
						xtra		=> $xtra,
						reqLevel	=> $reqLevel,
					};
		$cmdcnt++;
	};
	$sth->finish();
	&logPrint("Loaded $cmdcnt commands.\n");
	$main::stats_ref->{ commands } = $cmdcnt;
	$dbh->disconnect;
};

sub loadQuests {
	use DBD::SQLite;
	use FreezeThaw;
	my $dbh = DBI->connect("dbi:SQLite:dbname=$main::dbfile","",""); 
	my ($id, $shortDesc, $xp, $coins, $memoryData, $longDesc, $oldCode);
	my $query = 'SELECT id, shortDesc, xp, coins, memoryData, longDesc, oldCode FROM questData';
	my $sth  = $dbh->prepare($query) 
		or die "prepare: " . $dbh->errstr();
	$sth->execute() 
		or die "execute: " . $dbh->errstr();
	$sth->bind_columns(\$id, \$shortDesc, \$xp, \$coins, \$memoryData, \$longDesc, \$oldCode);
	my $qcnt = 0;
	while ($sth->fetch) {
		$main::questData{$id} = {
			id 		=> $id,
			shortDesc	=> $shortDesc,
			xp		=> $xp,
			coins		=> $coins,
			memoryData	=> $memoryData,
			longDesc	=> $longDesc,
			oldCode		=> $oldCode,
		};
		if ($memoryData ne '') {
			# Use stored memory data structure
			my @tmp = FreezeThaw::thaw $memoryData;
			$main::questData{$id}->{ memory } = \%{ $tmp[0] };
		} else {
			# Initialize memory data structure
			my %tmphash = ();
			$main::questData{$id}->{ memory } = \%tmphash;
		}; 
		$qcnt++;
	};
	$sth->finish();
	&logPrint("Loaded $qcnt quests.\n");
	$main::stats_ref->{ quests } = $qcnt;
	$dbh->disconnect;
};

sub loadNpcProto {
	use DBD::SQLite;
	my $dbh = DBI->connect("dbi:SQLite:dbname=$main::dbfile","",""); 
	my ($uid, $username, $desc, $level, $coins, $xp, $room, $godMode, $email, $gender, $paralyzed, $invisible, $invincible, $blind, $hitPoints, $mana, $maxHp, $maxMana, $lastCon, $lastDiscon, $lastSave, $npc, $npc_code_ref, $npc_loop_ref, $npc_loop_secs, $npc_code, $revivalTime, $rsp_rooms, $walk_rooms, $inst_limit, $poisoned, $intoxicated, $fuel, $memory);
	my $query = "SELECT UID, username, description, level, coins, XP, room, godmode, email, gender, paralyzed, invisible, invincible, blind, hitPoints, mana, maxHp, maxMana, lastCon, lastDiscon, lastSave, npc, npc_code_ref, npc_loop_ref, npc_loop_secs, npc_code, revivalTime, rsp_rooms, walk_rooms, inst_limit, poisoned, intoxicated, fuel, memory FROM npcProto";
	my $sth  = $dbh->prepare($query) 
		or die "prepare: " . $dbh->errstr();;
	$sth->execute() 
		or die "execute: " . $dbh->errstr();;
	$sth->bind_columns(\$uid, \$username, \$desc, \$level, \$coins, \$xp, \$room, \$godMode, \$email, \$gender, \$paralyzed, \$invisible, \$invincible, \$blind, \$hitPoints, \$mana, \$maxHp, \$maxMana, \$lastCon, \$lastDiscon, \$lastSave, \$npc, \$npc_code_ref, \$npc_loop_ref, \$npc_loop_secs, \$npc_code, \$revivalTime, \$rsp_rooms, \$walk_rooms, \$inst_limit, \$poisoned, \$intoxicated, \$fuel, \$memory)
        	or die "bind_columns: " . $dbh->errstr();
	my $protonpccntr = 0;
	my $protocodenpccntr = 0;
	my $protoloopnpccntr = 0;
	while ($sth->fetch) {
		my $tmpPresentItems_ref = {};
		# Store in array of hashes:
		$main::npcProto[$uid] = { 	username 	=> $username,
						desc	 	=> $desc,
						level	 	=> $level,
						coins	 	=> $coins,
						xp	 	=> $xp,
						room	 	=> $room,
						pid	 	=> -2,		# Not logged in.
						qid	 	=> -2,		# Not logged in.
						timeout	 	=> 0,		# Inactivity counter.
						godMode	 	=> $godMode,
						email	 	=> $email,
						gender	 	=> $gender,
						paralyzed 	=> $paralyzed,
						invisible 	=> $invisible,
						invincible 	=> $invincible,		# No fights possible at all.
						blind	 	=> $blind,
						hitPoints 	=> $hitPoints,
						mana	 	=> $mana,
						maxHp 		=> $maxHp,
						maxMana	 	=> $maxMana,
						lastCon	 	=> $lastCon,
						lastDiscon	=> $lastDiscon,
						lastSave	=> $lastSave,		
						inventory	=> $tmpPresentItems_ref,# User's inventory
						npc		=> $npc,		# If true, is NPC
						npc_code_ref	=> $npc_code_ref,	# NPC main sub ref
						npc_loop_ref	=> $npc_loop_ref,	# NPC loop sub ref
						npc_loop_secs	=> $npc_loop_secs,	# Period of NPC loop
						npc_code	=> $npc_code,		# NPC code filename
						revivalTime	=> $revivalTime,	# NPC revival period.
						rsp_rooms	=> $rsp_rooms,		# Respawn only here.
						walk_rooms	=> $walk_rooms,		# Walk only here.
						inst_limit	=> $inst_limit,		# Instances limit.
						poisoned	=> $poisoned,		# 
						intoxicated	=> $intoxicated,	# 
						fuel		=> $fuel,		# 
						memory		=> $memory,		# 
					};
		$protonpccntr++; 
		$protocodenpccntr++ if $npc_code ne ''; 
		$protoloopnpccntr++ if $npc_loop_ref ne ''; 
	};
	$sth->finish();
	&logPrint("Loaded $protonpccntr NPC prototypes ($protocodenpccntr coded, $protoloopnpccntr loop).\n");
	$main::stats_ref->{ protoNpcCount } = $protonpccntr;
	$main::stats_ref->{ protoCodeNpcCount } = $protocodenpccntr;
	$main::stats_ref->{ protoLoopnpcCount } = $protoloopnpccntr;
	$dbh->disconnect;
};

sub saveSingleUser {
	# dbh and user uid as parameter.
	my ($dbh, $uid, $noItems, undef) = @_;
	use FreezeThaw;
	$main::usersMainData[ $uid ]->{ lastSave } = DateTime::Format::SQLite->format_datetime(DateTime->now);
	my $memory = FreezeThaw::freeze $main::usersMainData[ $uid ]->{ memory };
	my $query = 'UPDATE mainUserData SET
			username = ?, password = ?, description = ?,
			level = ?, coins = ?, XP = ?, room = ?,
			godmode = ?, email = ?, gender = ?, paralyzed = ?,
			invisible = ?, invincible = ?, blind = ?, hitPoints = ?,
			mana = ?, maxHp = ?, maxMana = ?, lastCon = ?, 
			lastDiscon = ?, lastSave = ?, poisoned = ?,
			intoxicated = ?, fuel = ?, memory = ?, alignement = ?
			WHERE UID = ?';
	my $sth = $dbh->prepare($query)
		or die "prepare: " . $dbh->errstr();
	$sth->execute(
		$main::usersMainData[ $uid ]->{ username },
		$main::usersMainData[ $uid ]->{ password },
		$main::usersMainData[ $uid ]->{ desc },
		$main::usersMainData[ $uid ]->{ level },
		$main::usersMainData[ $uid ]->{ coins },
		$main::usersMainData[ $uid ]->{ xp },
		$main::usersMainData[ $uid ]->{ room },
		$main::usersMainData[ $uid ]->{ godMode },
		$main::usersMainData[ $uid ]->{ email },
		$main::usersMainData[ $uid ]->{ gender },
		$main::usersMainData[ $uid ]->{ paralyzed },
		$main::usersMainData[ $uid ]->{ invisible },
		$main::usersMainData[ $uid ]->{ invincible },
		$main::usersMainData[ $uid ]->{ blind },
		$main::usersMainData[ $uid ]->{ hitPoints },
		$main::usersMainData[ $uid ]->{ mana },
		$main::usersMainData[ $uid ]->{ maxHp },
		$main::usersMainData[ $uid ]->{ maxMana },
		$main::usersMainData[ $uid ]->{ lastCon },
		$main::usersMainData[ $uid ]->{ lastDiscon },
		$main::usersMainData[ $uid ]->{ lastSave },
		$main::usersMainData[ $uid ]->{ poisoned },
		$main::usersMainData[ $uid ]->{ intoxicated },
		$main::usersMainData[ $uid ]->{ fuel },
		$memory,
		$main::usersMainData[ $uid ]->{ alignement },
		$uid) 
		or die "execute: " . $dbh->errstr();
	$sth->finish;

	if ($noItems) { return; }; 	# When called from saveAllUsers, we don't need to save
					# inventory and solved quests because all items and quests 
					# will normally be saved next.
	# Save solved quests;
	$query = 'DELETE FROM userQuests WHERE uid = ?';
	$sth = $dbh->prepare($query)
		or die "prepare: " . $dbh->errstr();
	$sth->execute($uid)
		or die "execute: " . $dbh->errstr();
	$query = 'INSERT INTO userQuests(uid, qid) VALUES(?,?)';
	$sth = $dbh->prepare($query)
		or die "prepare: " . $dbh->errstr();
	foreach my $qid (keys %{ $main::usersMainData[ $uid ]->{ quests } } ) {
		my $qqid = $main::usersMainData[$uid]->{ quests }->{ $qid };
		$sth->execute($uid, $qqid)
			or die "execute: " . $dbh->errstr();
	};
	$sth->finish;
	# Save user's inventory:
	my $inv_ref = $main::usersMainData[ $uid ]->{ inventory };
	# Use delete and insert to speed up things and 
	# save newly created items:
	my $where_in = '-1';
	foreach my $iid (keys %$inv_ref) {
		$where_in = "$where_in,$iid";
	};
	$query = "DELETE FROM itemData WHERE id IN ($where_in)";
	$sth = $dbh->prepare($query)
		or die "prepare: " . $dbh->errstr();
	$sth->execute
		or die "execute: " . $dbh->errstr();
	$query = "INSERT INTO itemData(id, shortDesc, longDesc, gen_item_type, wearable, wieldable, movable, invisible, weight, volume, containVolume, dropable, created_by, edible, drinkable, key, missile, missile_type, missile_launcher, missile_type_needed, executable, code_ref, loop_ref, loop_secs, armourClass, weaponClass, amount, roomID, userID, itemID, worn, wielded, keepOnDeath, poison, healing, intox, questCode) VALUES(?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
	$sth = $dbh->prepare($query)
		or die "prepare: " . $dbh->errstr();
	foreach my $iid (keys %$inv_ref) {
		$sth->execute (
			$inv_ref->{ $iid }->{ id },
			$inv_ref->{ $iid }->{ shortDesc },
			$inv_ref->{ $iid }->{ longDesc },
			$inv_ref->{ $iid }->{ gen_item_type },
			$inv_ref->{ $iid }->{ wearable },
			$inv_ref->{ $iid }->{ wieldable },
			$inv_ref->{ $iid }->{ movable },
			$inv_ref->{ $iid }->{ invisible },
			$inv_ref->{ $iid }->{ weight },
			$inv_ref->{ $iid }->{ volume },
			$inv_ref->{ $iid }->{ containVolume },
			$inv_ref->{ $iid }->{ dropable },
			$inv_ref->{ $iid }->{ created_by },
			$inv_ref->{ $iid }->{ edible },
			$inv_ref->{ $iid }->{ drinkable },
			$inv_ref->{ $iid }->{ key },
			$inv_ref->{ $iid }->{ missile },
			$inv_ref->{ $iid }->{ missile_type },
			$inv_ref->{ $iid }->{ missile_launcher },
			$inv_ref->{ $iid }->{ missile_type_needed },
			$inv_ref->{ $iid }->{ executable },
			$inv_ref->{ $iid }->{ code_ref },
			$inv_ref->{ $iid }->{ loop_ref },
			$inv_ref->{ $iid }->{ loop_secs },
			$inv_ref->{ $iid }->{ armourClass },
			$inv_ref->{ $iid }->{ weaponClass },
			$inv_ref->{ $iid }->{ amount },
			$inv_ref->{ $iid }->{ roomID },
			$inv_ref->{ $iid }->{ userID },
			$inv_ref->{ $iid }->{ itemID },
			$inv_ref->{ $iid }->{ worn },
			$inv_ref->{ $iid }->{ wielded },
			$inv_ref->{ $iid }->{ keepOnDeath },
			$inv_ref->{ $iid }->{ poison },
			$inv_ref->{ $iid }->{ healing },
			$inv_ref->{ $iid }->{ intox },
			$inv_ref->{ $iid }->{ questCode },
			)
		or die "execute: " . $dbh->errstr();
	};
	$sth->finish;
};

sub saveAllItems {
	my $cnt = 0;
	use DBD::SQLite;
	my $dbh = DBI->connect("dbi:SQLite:dbname=$main::dbfile","",""); 
	# Use delete and insert in a transaction to speed up things and 
	# save newly created items:
	$dbh->begin_work;
	my $query = "DELETE FROM itemData";
	my $sth = $dbh->prepare($query)
		or die "prepare: " . $dbh->errstr();
	$sth->execute
		or die "execute: " . $dbh->errstr();
	$query = "INSERT INTO itemData(id, shortDesc, longDesc, gen_item_type, wearable, wieldable, movable, invisible, weight, volume, containVolume, dropable, created_by, edible, drinkable, key, missile, missile_type, missile_launcher, missile_type_needed, executable, code_ref, loop_ref, loop_secs, armourClass, weaponClass, amount, roomID, userID, itemID, worn, wielded, keepOnDeath, poison, healing, intox, questCode) VALUES(?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
	$sth = $dbh->prepare($query)
		or die "prepare: " . $dbh->errstr();
	foreach my $iid (keys %main::itemData) {
		$sth->execute (
			$main::itemData{ $iid }->{ id },
			$main::itemData{ $iid }->{ shortDesc },
			$main::itemData{ $iid }->{ longDesc },
			$main::itemData{ $iid }->{ gen_item_type },
			$main::itemData{ $iid }->{ wearable },
			$main::itemData{ $iid }->{ wieldable },
			$main::itemData{ $iid }->{ movable },
			$main::itemData{ $iid }->{ invisible },
			$main::itemData{ $iid }->{ weight },
			$main::itemData{ $iid }->{ volume },
			$main::itemData{ $iid }->{ containVolume },
			$main::itemData{ $iid }->{ dropable },
			$main::itemData{ $iid }->{ created_by },
			$main::itemData{ $iid }->{ edible },
			$main::itemData{ $iid }->{ drinkable },
			$main::itemData{ $iid }->{ key },
			$main::itemData{ $iid }->{ missile },
			$main::itemData{ $iid }->{ missile_type },
			$main::itemData{ $iid }->{ missile_launcher },
			$main::itemData{ $iid }->{ missile_type_needed },
			$main::itemData{ $iid }->{ executable },
			$main::itemData{ $iid }->{ code_ref },
			$main::itemData{ $iid }->{ loop_ref },
			$main::itemData{ $iid }->{ loop_secs },
			$main::itemData{ $iid }->{ armourClass },
			$main::itemData{ $iid }->{ weaponClass },
			$main::itemData{ $iid }->{ amount },
			$main::itemData{ $iid }->{ roomID },
			$main::itemData{ $iid }->{ userID },
			$main::itemData{ $iid }->{ itemID },
			$main::itemData{ $iid }->{ worn },
			$main::itemData{ $iid }->{ wielded },
			$main::itemData{ $iid }->{ keepOnDeath },
			$main::itemData{ $iid }->{ poison },
			$main::itemData{ $iid }->{ healing },
			$main::itemData{ $iid }->{ intox },
			$main::itemData{ $iid }->{ questCode },
			)
		or die "execute: " . $dbh->errstr();
		$cnt++;
	};
	$sth->finish;
	$dbh->commit;
	$dbh->disconnect;
	&logPrint("Saved $cnt items.\n");
};

sub saveAllUsers {
	my $cnt = 0;
	use DBD::SQLite;
	my $dbh = DBI->connect("dbi:SQLite:dbname=$main::dbfile","",""); 
	$dbh->begin_work;
	foreach my $lc_uid (keys @main::usersMainData) {
		if (defined $main::usersMainData[$lc_uid]) {
			#print "DEBUG: Saving " . $main::usersMainData[$lc_uid]->{ username } . " ($lc_uid)\n";
			&saveSingleUser($dbh, $lc_uid, 1); 	# Don't save inventory and quests.
			$cnt++;
		};
	};
	$dbh->commit;
	$dbh->disconnect;
	&logPrint("Saved $cnt users and NPCs.\n");
};

sub saveLoggedInUsers {
	my $cnt = 0;
	use DBD::SQLite;
	my $dbh = DBI->connect("dbi:SQLite:dbname=$main::dbfile","",""); 
	$dbh->begin_work;
	foreach my $username (@main::loggedInUsers) {
		my $l_uid = $main::userPass_ref->{$username}->{uid};
		#print "Saving $username ($l_uid)\n";
		&saveSingleUser($dbh, $l_uid);
		$cnt++;
	};
	$dbh->commit;
	$dbh->disconnect;
	if ($cnt) {
		&logPrint("Saved $cnt logged in users.\n");
	};
};

sub reloadItemLoopCode {
	my ($pid, undef) = @_;
	my ($username, $uid, $cmqid, $roomNum, $gender) = &intPid2Details($pid);

	# Go through all items, remove all notifiers and restart them with new code. 
	# Actually reload all item loop code. Item command code is loaded on the fly.
	use DBD::SQLite;
	my $dbh = DBI->connect("dbi:SQLite:dbname=$main::dbfile","",""); 
	my ($id, $executable, $loop_ref, $loop_secs);
	my $query = 'SELECT id, executable, loop_ref, loop_secs FROM itemData';
	my $sth  = $dbh->prepare($query) 
		or die "prepare: " . $dbh->errstr();;
	$sth->execute() 
		or die "execute: " . $dbh->errstr();;
	$sth->bind_columns(\$id, \$executable, \$loop_ref, \$loop_secs);
	my $itemcntr = 0;
	while ($sth->fetch) {
		# Main item data structure, used for mass saves:
		# Check for notifier code and add it to main loop:
		if (($loop_ref ne '') and ($loop_secs > 0)) {
			no strict "refs";
			# Remove old notifier:
			my $notifier_name = "ITEM_$id";
			my @notifiers = $main::loop->notifiers;
			foreach my $notif (@notifiers) {
				if ($notif->notifier_name eq $notifier_name) {
					$main::loop->remove($notif);
				};
			};
			# Load new code:
			delete $INC{ $executable };
			my $code = "require '$executable'";
			eval $code;
			if ($@) {
				&logPrint("Notifier error for item #$id: $@\n");
			} else {
				my $notifier_ref = IO::Async::Timer::Periodic->new(
					interval 	=>	$loop_secs,
					notifier_name	=>	"ITEM_$id",
					on_tick		=>	\&{ $loop_ref }
				);
				$notifier_ref->start;
				$main::loop->add($notifier_ref);
				$itemcntr++;
			};
		};
	};
	$sth->finish();
	if ($main::usersMainData[$uid]->{ invisible } == 0) {
		&informRoom($roomNum, $uid, "$username casts a spell and some objects glow!\n");
	} else {
		&informRoom($roomNum, $uid, "Somebody casts a spell and some objects glow!\n");
	};
	&clientTell($cmqid, $username, $pid, "Reloaded $itemcntr item loops.\n");
	&logPrint("Reloaded $itemcntr item loops.\n");
	$dbh->disconnect;
};

sub mainFightTick {
	# Go through @fightData and execute hits. 
	# fightData is an array of hashes with the following keys:
	# attUid: Attacker's uid.
	# defUid: Defender's uid.
	# timeOut: How long this fight will last when participants
	# 	   are away.
	# timeCnt: Counts the above. Reset on every hit.
	FIGHT:	foreach my $fight (keys %main::fightData) {
		if (not defined $main::fightData{$fight}->{ valid }) { 
			delete $main::fightData{$fight};
			&logPrint("Warning: Deleted undefined fight $fight from hash.\n");
			next FIGHT; 
		};
		if ($main::fightData{$fight}->{ valid } == 0) { next FIGHT; };
		my $attUid = $main::fightData{$fight}->{ attUid };
		my $defUid = $main::fightData{$fight}->{ defUid };
		my $timeOut = $main::fightData{$fight}->{ timeOut };
		my $timeCnt = $main::fightData{$fight}->{ timeCnt };

		my ($attusername, $attpid, $attcmqid, $attroomNum, $attgender) = &intUid2Details($attUid);
		my ($defusername, $defpid, $defcmqid, $defroomNum, $defgender) = &intUid2Details($defUid);

		# Check if participants are in the same room:
		if ($attroomNum == $defroomNum) {
			my $roomNum = $attroomNum;
			# In the same room, calculate hits.
			# Reset counter:
			$main::fightData{$fight}->{ timeCnt } = $timeOut;
			# Attacker hits first:
			my $damage = &intCalcHit($attUid, $defUid);
			my $damPct = ($damage*100)/$main::usersMainData[ $defUid ]->{ hitPoints };
			$main::usersMainData[ $defUid ]->{ hitPoints } -= $damage;
			$main::fightData{$fight}->{ attDmg } += $damage;
			# Inform everybody about fight:
			&intHitInfo($attUid, $defUid, $damPct, $roomNum);
			# Death/End of Fight check:
			if (&intEOFightCheck($attUid, $defUid, $fight)) { next FIGHT; };
			# Defender's turn:
			$damage = &intCalcHit($defUid, $attUid);
			$damPct = ($damage*100)/$main::usersMainData[ $attUid ]->{ hitPoints };
			$main::usersMainData[ $attUid ]->{ hitPoints } -= $damage;
			$main::fightData{$fight}->{ defDmg } += $damage;
			# Inform everybody about fight:
			&intHitInfo($defUid, $attUid, $damPct, $roomNum);
			# Death/End of Fight check:
			if (&intEOFightCheck($defUid, $attUid, $fight)) { next FIGHT; };
		} else {
			# Not in the same room, decrease timeCnt and return.
			# If timeCnt < 1, delete fight.
			$timeCnt--;
			if ($timeCnt > 0) {
				$main::fightData{$fight}->{ timeCnt } = $timeCnt;
			} else {
				if (not $main::usersMainData[ $attUid ]->{ npc }) {
					&clientTell($attcmqid, $attusername, $attpid, "You are not fighting $defusername anymore.\n");
				};
				if (not $main::usersMainData[ $defUid ]->{ npc }) {
					&clientTell($defcmqid, $defusername, $defpid, "You are not fighting $attusername anymore.\n");
				};
				# Instead of deleting, ivalidate fight. Invalid fights will be 
				# deleted before leaving the routine.
				$main::fightData{$fight}->{ valid } = 0;
			};
		};
	};
	# Now delete invalid fights from hash:
	foreach my $fight (keys %main::fightData) {
		if ($main::fightData{$fight}->{ valid } == 0) {
			#&logPrint("DEBUG: mainFightTick: Deleting fight $fight\n");
			delete $main::fightData{$fight};
		};
	};
};

sub saveNewUser {
	# user uid as parameter.
	# Should be called when a new user or NPC is created.
	my ($uid, undef) = @_;
	use DBD::SQLite;
	my $dbh = DBI->connect("dbi:SQLite:dbname=$main::dbfile","",""); 
	$main::usersMainData[ $uid ]->{ lastSave } = DateTime::Format::SQLite->format_datetime(DateTime->now);
	my $query = 'INSERT INTO mainUserData(uid, username, password, description, level, coins, XP, room,
			godmode, email, gender, paralyzed, invisible, invincible, blind, hitPoints,
			mana, maxHp, maxMana, lastCon, lastDiscon, lastSave, npc, npc_code_ref,
			npc_loop_ref, npc_loop_secs, npc_code, protoId, poisoned, intoxicated, fuel) 
			VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)';
	my $sth = $dbh->prepare($query)
		or die "prepare: " . $dbh->errstr();
	$sth->execute(
		$uid,
		$main::usersMainData[ $uid ]->{ username },
		$main::usersMainData[ $uid ]->{ password },
		$main::usersMainData[ $uid ]->{ desc },
		$main::usersMainData[ $uid ]->{ level },
		$main::usersMainData[ $uid ]->{ coins },
		$main::usersMainData[ $uid ]->{ xp },
		$main::usersMainData[ $uid ]->{ room },
		$main::usersMainData[ $uid ]->{ godMode },
		$main::usersMainData[ $uid ]->{ email },
		$main::usersMainData[ $uid ]->{ gender },
		$main::usersMainData[ $uid ]->{ paralyzed },
		$main::usersMainData[ $uid ]->{ invisible },
		$main::usersMainData[ $uid ]->{ invincible },
		$main::usersMainData[ $uid ]->{ blind },
		$main::usersMainData[ $uid ]->{ hitPoints },
		$main::usersMainData[ $uid ]->{ mana },
		$main::usersMainData[ $uid ]->{ maxHp },
		$main::usersMainData[ $uid ]->{ maxMana },
		$main::usersMainData[ $uid ]->{ lastCon },
		$main::usersMainData[ $uid ]->{ lastDiscon },
		$main::usersMainData[ $uid ]->{ lastSave },
		$main::usersMainData[ $uid ]->{ npc },
		$main::usersMainData[ $uid ]->{ npc_code_ref },
		$main::usersMainData[ $uid ]->{ npc_loop_ref },
		$main::usersMainData[ $uid ]->{ npc_loop_secs },
		$main::usersMainData[ $uid ]->{ npc_code },
		$main::usersMainData[ $uid ]->{ protoId },
		$main::usersMainData[ $uid ]->{ poisoned },
		$main::usersMainData[ $uid ]->{ intoxicated },
		$main::usersMainData[ $uid ]->{ fuel }
		) 
		or die "execute: " . $dbh->errstr();
	$sth->finish;
	$dbh->disconnect;
};

sub intEOFightCheck {
	# Check for death or end of fight. Return 1 if EOF, 0 otherwise.
	# All death code (exp, coins, items) goes here.
	my ($attUid, $defUid, $fight, undef) = @_;
	my ($attusername, $attpid, $attcmqid, $attroomNum, $attgender) = &intUid2Details($attUid);
	my ($defusername, $defpid, $defcmqid, $defroomNum, $defgender) = &intUid2Details($defUid);
	if (
		($main::usersMainData[ $defUid ]->{ hitPoints } < 1) and
		($main::usersMainData[ $defUid ]->{ npc } == 0)
	) {
		# Real player.
		# No death yet, death routines missing... :-)
		$main::usersMainData[ $defUid ]->{ hitPoints } = 1;
		# Stop fight:
		&userTell($attcmqid, $attusername, $attpid, $attUid, "$defusername kneels before you!\n");
		&userTell($defcmqid, $defusername, $defpid, $defUid, "You kneel before $attusername!\n");
		my $attusername2 = 'Somebody';
		my $defusername2 = 'Somebody';
		if ($main::usersMainData[$attUid]->{ invisible } == 0) {
			$attusername2 = $attusername;
		};
		if ($main::usersMainData[$defUid]->{ invisible } == 0) {
			$defusername2 = $defusername;
		};
			&informRoom($attroomNum, $attUid, "$defusername2 kneels before $attusername2!\n", $defUid);
		#delete $main::fightData{$fight};
		$main::fightData{$fight}->{ valid } = 0;
		return 1;
	};
	if (
		($main::usersMainData[ $defUid ]->{ hitPoints } < 1) and
		($main::usersMainData[ $defUid ]->{ npc } == 1)
	) {
		# NPCs. These are mortal.
		$main::usersMainData[ $defUid ]->{ room } = -10;
		$main::usersMainData[ $defUid ]->{ revivalCounter } = $main::usersMainData[ $defUid ]->{ revivalTime };
		push @main::deadNpcs, $defUid;
		# Remove NPC from room:
		my( $index )= grep { ${ $main::roomsData[ $defroomNum ]->{ presentUids } }[$_] eq $defUid } 0..$#{ $main::roomsData[ $defroomNum ]->{ presentUids } };
		splice(@{ $main::roomsData[ $defroomNum ]->{ presentUids } }, $index, 1);
		my( $indexu )= grep { ${ $main::roomsData[ $defroomNum ]->{ presentUsers } }[$_] eq $defusername } 0..$#{ $main::roomsData[ $defroomNum ]->{ presentUsers } };
		splice(@{ $main::roomsData[ $defroomNum ]->{ presentUsers } }, $indexu, 1);
		# Stop fight:
		my $attusername2 = 'Somebody';
		my $defusername2 = 'Somebody';
		if ($main::usersMainData[$attUid]->{ invisible } == 0) {
			$attusername2 = $attusername;
		};
		if ($main::usersMainData[$defUid]->{ invisible } == 0) {
			$defusername2 = $defusername;
		};
		&userTell($attcmqid, $attusername, $attpid, $attUid, "$defusername2 dies!\n");
		&informRoom($defroomNum, $attUid, "$attusername2 killed $defusername2!\n", $defUid);
		# Locate all fights of $defUid and delete them. Keep uid and dmg of opponents
		# to share XP.
		my %otherOpp = ();
		my @todel = ();
		# Also store killer's dmg before deleting fight:
		my $attDmg = $main::fightData{$fight}->{ attDmg };
		my $totalDmg = $attDmg;
		SRCHF:	foreach my $lf (keys %main::fightData) {
			if ($lf eq $fight) { next SRCHF; };
			if ($main::fightData{$lf}->{ valid } == 0) { next SRCHF; };
			my $ouid;
			my $dmg;
			if ($main::fightData{$lf}->{ attUid } == $defUid) {
				$ouid = $main::fightData{$lf}->{ defUid };
				$dmg = $main::fightData{$lf}->{ defDmg };
			} else {
				$ouid = $main::fightData{$lf}->{ attUid };
				$dmg = $main::fightData{$lf}->{ attDmg };
			};
			$otherOpp{ $ouid } = $dmg;
			$totalDmg += $dmg;
			if (not $main::usersMainData[ $ouid ]->{ npc }) {
				my ($ousername, $opid, $ocmqid, $oroomNum, $ogender) = &intUid2Details($ouid);
				&clientTell($ocmqid, $ousername, $opid, "You are not fighting $defusername anymore.\n");
			};
			push @todel, $lf;
		};
		$main::fightData{$fight}->{ valid } = 0;
		#delete $main::fightData{$fight};
		foreach my $lf (@todel) {
			#delete $main::fightData{$lf};
			$main::fightData{$lf}->{ valid } = 0;
		};

		# Experience to player. NPCs XP field holds the XP gain of the player.
		# Use stored damage points to calculate percentage of XP for each 
		# player in fight with the victim in time of death.
		# This works normally while NPCs don't regenerate. If NPCs regeneration
		# is implemented, will be a mess.
		my $attXpGain = $main::usersMainData[ $defUid ]->{ xp } * ($attDmg/$totalDmg);
		$attXpGain = (sprintf("%.0f", $attXpGain)) + 0;
		if ($attXpGain < 1) { $attXpGain = 1; };
		$main::usersMainData[ $attUid ]->{ xp } += $attXpGain;
		&userTell($attcmqid, $attusername, $attpid, $attUid, "You feel $attXpGain point(s) more experienced.\n");
		# Now loop through %otherOpp:
		foreach my $ouid (keys %otherOpp) {
			my $dmg = $otherOpp{$ouid};
			#print "DEBUG: $ouid Dmg = $dmg\n";
			my $xpGain = $main::usersMainData[ $defUid ]->{ xp } * ($dmg/$totalDmg);
			$xpGain = (sprintf("%.0f", $xpGain)) + 0;
			if ($xpGain < 1) { $xpGain = 1; };
			$main::usersMainData[ $ouid ]->{ xp } += $xpGain;
			my ($ousername, $opid, $ocmqid, $oroomNum, $ogender) = &intUid2Details($ouid);
			&userTell($ocmqid, $ousername, $opid, $ouid, "You feel $xpGain point(s) more experienced.\n");
		};
		# Money to room. Create a coin item based on npc's coins. 
		my $rc = &intCreateMoney($defroomNum, $main::usersMainData[ $defUid ]->{ coins });
		if ($rc) {
			&informRoom($defroomNum, -1, "$defusername2 drops some coins.\n");
		};
		# Inventory to room.
		my $dropped = &intInv2Room($defUid, $defroomNum); 
		if ($dropped) {
			&informRoom($defroomNum, -1, "$defusername2 drops some stuff.\n");
		};
		return 1;
	};
	# No EOF:
	return 0;
};

sub intInv2Room {
	# Dump a uid's inventory on the floor.
	my ($uid, $roomNum, undef) = @_;
	my $cnt = 0;
	my $inv_ref = $main::usersMainData[ $uid ]->{ inventory };
	foreach my $iid (keys %$inv_ref) {
		if ($inv_ref->{ $iid }->{ dropable } == 1) { 	# Only dropable
			$cnt++;
			# Change main data structure:
			$main::itemData{ $iid }->{ roomID } = $roomNum;
			$main::itemData{ $iid }->{ userID } = 0;
			# Change inventory data structure: 
			my $rec = $main::usersMainData[ $uid ]->{ inventory }->{ $iid };
			delete $main::usersMainData[ $uid ]->{ inventory }->{ $iid };
			# Change room data structure:
			$rec->{ roomID } = $roomNum;
			$rec->{ userID } = 0;
			$main::roomsData[ $roomNum ]->{ presentItems }->{ $iid } = $rec;
		};
	};
	return $cnt;
};

sub reloadNpcLoopCode {
	my ($pid, undef) = @_;
	my ($username, $xuid, $cmqid, $roomNum, $gender) = &intPid2Details($pid);

	# Go through all NPCs, remove all notifiers and restart them with new code. 
	# Actually reload all NPCs loop code. NPCs command code is loaded on the fly.
	use DBD::SQLite;
	my $dbh = DBI->connect("dbi:SQLite:dbname=$main::dbfile","",""); 
	my ($uid, $npc_code, $npc_loop_ref, $npc_loop_secs);
	my $query = 'SELECT uid, npc_code, npc_loop_ref, npc_loop_secs FROM mainUserData where npc = 1';
	my $sth  = $dbh->prepare($query) 
		or die "prepare: " . $dbh->errstr();;
	$sth->execute() 
		or die "execute: " . $dbh->errstr();;
	$sth->bind_columns(\$uid, \$npc_code, \$npc_loop_ref, \$npc_loop_secs);
	my $npccntr = 0;
	while ($sth->fetch) {
		# Check for notifier code and add it to main loop:
		if (($npc_loop_ref ne '') and ($npc_loop_secs > 0) and ($npc_code ne '')) {
			no strict "refs";
			# Remove old notifier:
			my $notifier_name = "NPC_$uid";
			my @notifiers = $main::loop->notifiers;
			foreach my $notif (@notifiers) {
				if ($notif->notifier_name eq $notifier_name) {
					$main::loop->remove($notif);
				};
			};
			# Load new code:
			delete $INC{ $npc_code };
			my $code = "require '$npc_code'";
			eval $code;
			if ($@) {
				&logPrint("Notifier error for NPC #$uid: $@\n");
			} else {
				my $notifier_ref = IO::Async::Timer::Periodic->new(
					interval 	=>	$npc_loop_secs,
					notifier_name	=>	"NPC_$uid",
					on_tick		=>	\&{ $npc_loop_ref }
				);
				$notifier_ref->start;
				$main::loop->add($notifier_ref);
				$npccntr++;
			};
		};
	};
	$sth->finish();
	if ($main::usersMainData[$uid]->{ invisible } == 0) {
		&informRoom($roomNum, $xuid, "$username casts a spell and some monsters glow!\n");
	} else {
		&informRoom($roomNum, $xuid, "Somebody casts a spell and some monsters glow!\n");
	};
	&clientTell($cmqid, $username, $pid, "Reloaded $npccntr NPC loops.\n");
	&logPrint("Reloaded $npccntr NPC loops.\n");
	$dbh->disconnect;
};


sub saveAllQuests {
	my $cnt = 0;
	use DBD::SQLite;
	use FreezeThaw;
	my $dbh = DBI->connect("dbi:SQLite:dbname=$main::dbfile","",""); 
	# Use delete and insert in a transaction to speed up things and 
	# save newly created items:
	$dbh->begin_work;
	my $query = "DELETE FROM questData";
	my $sth = $dbh->prepare($query)
		or die "prepare: " . $dbh->errstr();
	$sth->execute
		or die "execute: " . $dbh->errstr();
	$query = "INSERT INTO questData(id, shortDesc, xp, coins, memoryData, longDesc, oldCode) VALUES(?, ?, ?, ?, ?, ?, ?)";
	$sth = $dbh->prepare($query)
		or die "prepare: " . $dbh->errstr();
	foreach my $qid (keys %main::questData) {
		my $rec = $main::questData{$qid};
		bless $rec;
		$rec->{ memoryData } = FreezeThaw::freeze $rec->{ memory };
		$sth->execute (
			$rec->{ id },
			$rec->{ shortDesc },
			$rec->{ xp },
			$rec->{ coins },
			$rec->{ memoryData },
			$rec->{ longDesc },
			$rec->{ oldCode },
			)
		or die "execute: " . $dbh->errstr();
		$cnt++;
	};
	$sth->finish;
	$dbh->commit;
	$dbh->disconnect;
	&logPrint("Saved $cnt quests.\n");
};

sub saveAllUsersQuests {
	# We need this to reset the autoincrement primary key in userQuests table.
	# Single user save is very frequent and will increment the primary key of
	# userQuests very fast.
	# This routine will be called whenever saveAllUsers is called. 
	my $cnt = 0;
	use DBD::SQLite;
	my $dbh = DBI->connect("dbi:SQLite:dbname=$main::dbfile","",""); 
	$dbh->begin_work;
	my $query = 'DELETE FROM userQuests';
	my $sth = $dbh->prepare($query)
		or die "prepare: " . $dbh->errstr();
	$sth->execute()
		or die "execute: " . $dbh->errstr();
	$query = 'INSERT INTO userQuests(uid, qid) VALUES(?,?)';
	$sth = $dbh->prepare($query)
		or die "prepare: " . $dbh->errstr();
	foreach my $lc_uid (keys @main::usersMainData) {
		if (defined $main::usersMainData[$lc_uid]) {
			foreach my $qid (keys %{ $main::usersMainData[ $lc_uid ]->{ quests } } ) {
				my $qqid = $main::usersMainData[$lc_uid]->{ quests }->{ $qid };
				$sth->execute($lc_uid, $qqid)
					or die "execute: " . $dbh->errstr();
				$cnt++;
			};
		};
	};
	$dbh->commit;
	$dbh->disconnect;
	&logPrint("Saved $cnt solved quests\' records.\n");
};

sub reloadItemCode {
	my ($pid, undef) = @_;
	# Go through all items and revalidate command code.
	use DBD::SQLite;
	my $dbh = DBI->connect("dbi:SQLite:dbname=$main::dbfile","",""); 
	my ($id, $executable, $code_ref);
	my $query = 'SELECT id, executable, code_ref FROM itemData';
	my $sth  = $dbh->prepare($query) 
		or die "prepare: " . $dbh->errstr();;
	$sth->execute() 
		or die "execute: " . $dbh->errstr();;
	$sth->bind_columns(\$id, \$executable, \$code_ref);
	my $itemcntr = 0;
	while ($sth->fetch) {
		# Check for command code and add it to main loop:
		if ($code_ref ne '') {
			no strict "refs";
			# Load new code:
			delete $INC{ $executable };
			my $code = "require '$executable'";
			eval $code;
			if ($@) {
				&logPrint("Command code error for item #$id: $@\n");
				$main::itemData{ $id }->{ validCode } = 0;
			} else {
				$main::itemData{ $id }->{ validCode } = 1;
				$itemcntr++;
			};
		};
	};
	$sth->finish();
	$dbh->disconnect;
	&logPrint("Reloaded $itemcntr item source files.\n");
	# Rest only on interactive call:
	if ((not defined $pid) or ($pid < 1)) { return; }; 
	my ($username, $uid, $cmqid, $roomNum, $gender) = &intPid2Details($pid);
	if ($main::usersMainData[$uid]->{ invisible } == 0) {
		&informRoom($roomNum, $uid, "$username casts a spell and some objects glow!\n");
	} else {
		&informRoom($roomNum, $uid, "Somebody casts a spell and some objects glow!\n");
	};
	&clientTell($cmqid, $username, $pid, "Reloaded $itemcntr item source files.\n");
};



sub reloadNpcCode {
	my ($pid, undef) = @_;

	# Go through all NPCs, remove all notifiers and restart them with new code. 
	# Actually reload all NPCs loop code. NPCs command code is loaded on the fly.
	use DBD::SQLite;
	my $dbh = DBI->connect("dbi:SQLite:dbname=$main::dbfile","",""); 
	my ($uid, $npc_code, $npc_code_ref);
	my $query = 'SELECT uid, npc_code, npc_code_ref FROM mainUserData where npc = 1';
	my $sth  = $dbh->prepare($query) 
		or die "prepare: " . $dbh->errstr();;
	$sth->execute() 
		or die "execute: " . $dbh->errstr();;
	$sth->bind_columns(\$uid, \$npc_code, \$npc_code_ref);
	my $npccntr = 0;
	while ($sth->fetch) {
		# Check for code :
		if (($npc_code_ref ne '') and ($npc_code ne '')) {
			no strict "refs";
			# Load new code:
			delete $INC{ $npc_code };
			my $code = "require '$npc_code'";
			eval $code;
			if ($@) {
				&logPrint("Code error for NPC #$uid: $@\n");
				$main::usersMainData[$uid]->{ validCode } = 0;
			} else {
				$npccntr++;
				$main::usersMainData[$uid]->{ validCode } = 1;
			};
		};
	};
	$sth->finish();
	$dbh->disconnect;
	&logPrint("Reloaded $npccntr NPC code files.\n");
	# Rest only on interactive call:
	if ((not defined $pid) or ($pid < 1)) { return; }; 
	my ($username, $xuid, $cmqid, $roomNum, $gender) = &intPid2Details($pid);
	if ($main::usersMainData[$uid]->{ invisible } == 0) {
		&informRoom($roomNum, $xuid, "$username casts a spell and some monsters glow!\n");
	} else {
		&informRoom($roomNum, $xuid, "Somebody casts a spell and some monsters glow!\n");
	};
	&clientTell($cmqid, $username, $pid, "Reloaded $npccntr NPC code files.\n");
};

sub intMetabolism {
	# Deal with fuel, poisons, intoxication etc. Must be called
	# from mainMudaTick, high frequency.
	foreach my $username (@main::loggedInUsers) {
		my $uid = $main::userPass_ref->{$username}->{uid};
		my (undef, $pid, $cmqid, $roomNum, $gender) = &intUid2Details($uid);
		my $eUnitsTotal = 0;
		if ($main::usersMainData[$uid]->{ invisible } > 0) {
			# Decrease invisibility units:
			$main::usersMainData[$uid]->{ invisible } -= 1;
			if ($main::usersMainData[$uid]->{ invisible } <= 0) {
				$main::usersMainData[$uid]->{ invisible } = 0;
				&clientTell($cmqid, $username, $pid, "You are not invisible anymore.\n");
				&informRoom($roomNum, $uid, "$username becomes visible again!\n");
			};
		};
		if ($main::usersMainData[$uid]->{ intoxicated } > 0) {
			# Healing by intoxication:
			$main::usersMainData[$uid]->{ intoxicated } -= 2;
			if ($main::usersMainData[$uid]->{ hitPoints } < $main::usersMainData[$uid]->{ maxHp }) {
				$main::usersMainData[$uid]->{ hitPoints } += 2;
			};
			if ($main::usersMainData[$uid]->{ hitPoints } > $main::usersMainData[$uid]->{ maxHp }) {
				$main::usersMainData[$uid]->{ hitPoints } = $main::usersMainData[$uid]->{ maxHp };
				&clientTell($cmqid, $username, $pid, "You feel completely healed.\n");
			};
			if ($main::usersMainData[$uid]->{ intoxicated } <= 0) {
				$main::usersMainData[$uid]->{ intoxicated } = 0;
				&clientTell($cmqid, $username, $pid, "You are not drunk anymore.\n");
			};
			$main::usersMainData[$uid]->{ intoxicated } = (sprintf("%.2f", $main::usersMainData[$uid]->{ intoxicated })) + 0;
			$main::usersMainData[$uid]->{ hitPoints } = (sprintf("%.2f", $main::usersMainData[$uid]->{ hitPoints })) + 0;
		};
		if ($main::usersMainData[$uid]->{ hitPoints } < $main::usersMainData[$uid]->{ maxHp }) {
			if ($main::usersMainData[$uid]->{ fuel } <= 0) {
				if (rand(100) <= 5) {
					&clientTell($cmqid, $username, $pid, "You are starving.\n");
				};
			};
			# We may need to rethink the following line.
			my $eUnit = ($main::usersMainData[$uid]->{ fuel } / 100);
			$main::usersMainData[$uid]->{ hitPoints } += $eUnit;
			if ($main::usersMainData[$uid]->{ hitPoints } > $main::usersMainData[$uid]->{ maxHp }) {
				$main::usersMainData[$uid]->{ hitPoints } = $main::usersMainData[$uid]->{ maxHp };
				&clientTell($cmqid, $username, $pid, "You feel completely healed.\n");
			};
			$main::usersMainData[$uid]->{ hitPoints } = (sprintf("%.2f", $main::usersMainData[$uid]->{ hitPoints })) + 0;
			$eUnitsTotal += $eUnit;
		};
		if ($main::usersMainData[$uid]->{ mana } < $main::usersMainData[$uid]->{ maxMana }) {
			if ($main::usersMainData[$uid]->{ fuel } <= 0) {
				if (rand(100) <= 5) {
					&clientTell($cmqid, $username, $pid, "You are starving.\n");
				};
			};
			# We may need to rethink the following line.
			my $eUnit = ($main::usersMainData[$uid]->{ fuel } / 100);
			$main::usersMainData[$uid]->{ mana } += $eUnit;
			if ($main::usersMainData[$uid]->{ mana } > $main::usersMainData[$uid]->{ maxMana }) {
				$main::usersMainData[$uid]->{ mana } = $main::usersMainData[$uid]->{ maxMana };
				&clientTell($cmqid, $username, $pid, "You feel your magic strong again.\n");
			};
			$main::usersMainData[$uid]->{ mana } = (sprintf("%.2f", $main::usersMainData[$uid]->{ mana })) + 0;
			$eUnitsTotal += $eUnit;
		};
		if ($main::usersMainData[$uid]->{ fuel } > 0) {
			# Again, needs adjustment. This value is almost random:
			if (($eUnitsTotal <= 0.30) and ($eUnitsTotal > 0)) { $eUnitsTotal = 0.30; };
			$main::usersMainData[$uid]->{ fuel } -= $eUnitsTotal*0.2;
			$main::usersMainData[$uid]->{ fuel } = (sprintf("%.2f", $main::usersMainData[$uid]->{ fuel })) + 0;
			if ($main::usersMainData[$uid]->{ fuel } <= 1.5) {
				$main::usersMainData[$uid]->{ fuel } = 0;
				&clientTell($cmqid, $username, $pid, "You are starving.\n");
			};
		};
	};
};

sub checkRCMDintegrity {
	use DBD::SQLite;
	my $dbh = DBI->connect("dbi:SQLite:dbname=$main::dbfile","",""); 
	my $errcnt = 0;
	my ($id, $shortDesc, $movable, $invisible, $executable, $code_ref, $loop_ref, $loop_secs, $roomID, $userID, $itemID);
	my $query = 'SELECT id, shortDesc, movable, invisible, executable, code_ref, loop_ref, loop_secs, roomID, userID, itemID FROM itemData WHERE shortDesc like \'RCMD%\'';
	my $sth  = $dbh->prepare($query) 
		or die "prepare: " . $dbh->errstr();;
	$sth->execute() 
		or die "execute: " . $dbh->errstr();;
	$sth->bind_columns(\$id, \$shortDesc, \$movable, \$invisible, \$executable, \$code_ref, \$loop_ref, \$loop_secs, \$roomID, \$userID, \$itemID);
	my %found = ();
	while ($sth->fetch) {
		my $error = 0;
		if ($movable) {
			&logPrint( "Integrity: $shortDesc (#$id) is movable.\n" );
			$error = 1;
		};
		if (not $invisible) {
			&logPrint( "Integrity: $shortDesc (#$id) is visible.\n" );
			$error = 1;
		};
		if (not $loop_secs) {
			&logPrint( "Warning: $shortDesc (#$id) has zero loop secs.\n" );
		};
		if ($userID) {
			&logPrint( "Integrity: $shortDesc (#$id) on user.\n" );
			$error = 1;
		};
		if ($itemID) {
			&logPrint( "Integrity: $shortDesc (#$id) in item.\n" );
			$error = 1;
		};
		# Find room num:
		my (undef, $roomNum) = split /RCMD/, $code_ref, 2;
		if (defined $found{$code_ref}) {
			&logPrint( "Integrity: $shortDesc (#$id) found in more rooms ($found{$code_ref}, $roomNum).\n" );
			$error = 1;
		};
		$found{$code_ref} = $roomNum;
		if ($roomNum ne $roomID) { 
			&logPrint( "Integrity: $shortDesc (#$id) found in wrong room ($roomNum).\n" );
			$error = 1;
		};
		if ($executable ne "itemCode/$code_ref.pl") {
			&logPrint( "Integrity: $shortDesc (#$id) has wrong filename ($executable).\n" );
			$error = 1;
		};
		if ($loop_ref ne ("$code_ref" . 'loop')) {
			&logPrint( "Integrity: $shortDesc (#$id) has wrong loop name ($loop_ref).\n" );
			$error = 1;
		};
		if ($error) { $errcnt++; };
	};
	if ($errcnt) {
		&logPrint( "RCMD integrity: found $errcnt errors.\n" );
	} else {
		&logPrint( "RCMD integrity check passed.\n" );
	};
};




END { } # module clean-up code here (global destructor)

1; # don't forget to return a true value from the file
