#############################################################################
# MUDA: Resurrection							    #
# Ms_Cmds.pm		: Standard user commands.			    #
# Artist:		: Theodore J. Soldatos				    #
#############################################################################
# This file is part of MUDA:Resurrection.			   	    #
#									    #
# MUDA:Resurrection is free software: you can redistribute it and/or modify #
# it under the terms of the GNU General Public License as published by	    #
# the Free Software Foundation, either version 3 of the License, or	    #
# (at your option) any later version.					    #
#									    #
# MUDA:Resurrection is distributed in the hope that it will be useful,	    #
# but WITHOUT ANY WARRANTY; without even the implied warranty of	    #
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the		    #
# GNU General Public License for more details.				    #
#									    #
# You should have received a copy of the GNU General Public License	    #
# along with MUDA:Resurrection.  If not, see <http://www.gnu.org/licenses/>.#
#									    #
# Copyright 2012 Theodore J. Soldatos					    #
#############################################################################


package Ms_Libs;

use strict;
use warnings;

use IPC::SysV qw(IPC_PRIVATE IPC_RMID IPC_CREAT S_IRWXU IPC_NOWAIT);
use Text::Autoformat;

BEGIN {
use Exporter ();
our ($VERSION, @ISA, @EXPORT, @EXPORT_OK, %EXPORT_TAGS);
# set the version for version checking
$VERSION = 1.00;
@ISA = qw(Exporter);
@EXPORT = qw(&cmdLook &cmdUnknown &cmdSay &cmdGo &cmdTeleport &cmdStatus &cmdHelp &cmdSet &cmdWho &cmdGet &cmdInventory &cmdDrop &cmdExamine &cmdWear &cmdTakeoff &cmdWield &cmdUnwield &cmdCreateItem &cmdDestroyItem &cmdGive &cmdShout &cmdHit &cmdSummon &cmdTell &cmdWhisper &cmdKiss &cmdHug &cmdHand &cmdShake &cmdSpit &cmdBite &cmdKick &cmdPush &cmdCaress &cmdTouch &cmdSmile &cmdLaugh &cmdBlush &cmdGrin &cmdQuests &cmdShoot &cmdAdvance &cmdEat &cmdDrink &intEatDrink);
%EXPORT_TAGS = ( ); # eg: TAG => [ qw!name1 name2! ],
#@EXPORT_OK = qw();
}
our @EXPORT_OK;

sub cmdLook {
	my ($pid, $clientArgs, undef) = @_;
	my ($username, $uid, $cmqid, $roomNum, $gender) = &intPid2Details($pid);
	if (not defined $clientArgs) { $clientArgs = ''; };
	# Check if AT:
	my $subcmd;
	($subcmd, $clientArgs) = split(/ /, $clientArgs, 2);
	if (not defined $subcmd) { $subcmd = ''; };
	if (not defined $clientArgs) { $clientArgs = ''; };
	if (uc($subcmd) eq 'AT') {
		my $found = 0;
		if (length($clientArgs) < 2) {
			&clientTell($cmqid, $username, $pid, "Please be more specific.\n");
			return;
		};
		# Look AT something. Check for persons or items.
		if (uc($username) eq uc($clientArgs)) {
			&clientTell($cmqid, $username, $pid, "You need a mirror for that.\n");
			return;
		};
		if ((scalar(@{ $main::roomsData[ $roomNum ]->{presentUids} }) > 0)
			or (scalar keys %{ $main::roomsData[ $roomNum ]->{ presentItems } } > 0))
		{
			# Persons.
			foreach my $ouid (@{ $main::roomsData[ $roomNum ]->{presentUids} }) {
				# Loop through present persons and look for a username match:
				#if (uc($main::usersMainData[$ouid]->{ username }) eq uc($clientArgs)) {
				if (&intCmpUid2name($ouid, $clientArgs)) {
					# Just the description for now.
					my $ist = "";
					if ($main::usersMainData[$ouid]->{ pid } == -1) {
						$ist = " the ice statue of";
					};
					# Actual user:
					my $msg = "This is$ist " . $main::usersMainData[$ouid]->{ username } . ", " . $main::usersMainData[$ouid]->{ desc } . ".\n";
					&clientTell($cmqid, $username, $pid, $msg);
					# User's inventory. Should only list wielded and worn items.
					my $v_inv_ref = $main::usersMainData[$ouid]->{ inventory };
					my (@wielded, @worn) = ();
					foreach my $viid (keys %$v_inv_ref) {
						if ($v_inv_ref->{ $viid }->{ worn } == 1) {
							push @worn, $v_inv_ref->{ $viid }->{ shortDesc };
						};
						if ($v_inv_ref->{ $viid }->{ wielded } == 1) {
							push @wielded, $v_inv_ref->{ $viid }->{ shortDesc };
						};
					};
					if (scalar(@worn) > 0) {
						my $msg = $main::usersMainData[$ouid]->{ username } . " wears:\n";
						&clientTell($cmqid, $username, $pid, $msg);
						foreach $msg (@worn) {
							&clientTell($cmqid, $username, $pid, "$msg.\n");
						};
					};
					if (scalar(@wielded) > 0) {
						my $msg = $main::usersMainData[$ouid]->{ username } . " wields:\n";
						&clientTell($cmqid, $username, $pid, $msg);
						foreach $msg (@wielded) {
							&clientTell($cmqid, $username, $pid, "$msg.\n");
						};
					};
					if ($main::usersMainData[$uid]->{ invisible } == 0) {
						# Room: 
						$msg = "$username looks at$ist " . $main::usersMainData[$ouid]->{ username } . ".\n";
						&informRoom($roomNum, $uid, $msg, $ouid);
						# Looked at:
						if ($main::usersMainData[$ouid]->{ pid } > 0) {
							&clientTell(	$main::usersMainData[$ouid]->{ qid },
									$main::usersMainData[$ouid]->{ username },
									$main::usersMainData[$ouid]->{ pid },
									"$username looks at you.\n");
						} elsif ($main::usersMainData[$ouid]->{ npc } == 1) {
							# Call interact routine of NPC, if available.
							&npcTell($pid, $ouid, '__LOOKEDAT', '');
						};
					};
					$found = 1;
				};
			};
			# Items.
			my $escClientArgs = quotemeta($clientArgs);
			foreach my $iid (keys %{ $main::roomsData[ $roomNum ]->{ presentItems } }) {
				my $item_ref = $main::itemData{ $iid };
				if ($item_ref->{ invisible } == 0) { 	# Only visible
					my $shortDesc = $item_ref->{ shortDesc };
					if ($shortDesc =~ m/$escClientArgs/i) { 
						my $formatted = autoformat $item_ref->{ longDesc }, { left=>1, right=>$main::termWidth, justify => 'full' };
						&clientTell($cmqid, $username, $pid, $formatted);
						#if ($item_ref->{ wearable } == 1) {
						#	&clientTell($cmqid, $username, $pid, "Armour class: " . $item_ref->{ armourClass } . ".\n");
						#};
						#if ($item_ref->{ wieldable } == 1) {
						#	&clientTell($cmqid, $username, $pid, "Weapon class: " . $item_ref->{ weaponClass } . ".\n");
						#};
						#if ($item_ref->{ executable } ne '') {
						#	&clientTell($cmqid, $username, $pid, "You can use this in more ways.\n");
						#};
						$found = 1;
						last;	# First match only.
					};
				};
			};
		} else {
			# Nothing found.
			&clientTell($cmqid, $username, $pid, "No $clientArgs here.\n");
		};
		if (not $found) {
			&clientTell($cmqid, $username, $pid, "No $clientArgs here.\n");
		};
		return;
	};
	# Print room:
	&clientPrintRoom($cmqid, $username, $pid);
	# Print people in room:
	if (scalar(@{ $main::roomsData[ $roomNum ]->{presentUids} }) > 0) {
		# There are people here.
		&clientTell($cmqid, $username, $pid, "         \n");

		foreach my $ouid (@{ $main::roomsData[ $roomNum ]->{presentUids} }) {
			if ($main::usersMainData[$ouid]->{ username } eq $username) {next;};
			if ($main::usersMainData[$ouid]->{ invisible }) {next;};
			my $msg = '';
			if ($main::usersMainData[$ouid]->{ pid }== -1) {
				# Statue:
				$msg = "The ice statue of " . $main::usersMainData[$ouid]->{ username }. " is here.\n";
			} else {
				$msg = $main::usersMainData[$ouid]->{ username } . " is here.\n";
			};
			&clientTell($cmqid, $username, $pid, $msg);
		};
		&clientTell($cmqid, $username, $pid, "         \n");
	}; 
	# Print items in room:
	my $item_ref = $main::roomsData[ $roomNum ]->{ presentItems };
	my @items = ();
	# Consolidate
	&intConsolidateInv($item_ref);
	foreach my $iid (keys %$item_ref) {
		if ($item_ref->{ $iid }->{ invisible } == 0) {
			push @items, $iid;
		};
	};
	if (scalar(@items) > 0) {
		# Visible Items in the room:
		&clientTell($cmqid, $username, $pid, "You can see:\n");
		foreach my $iid (@items) {
			my $amountStr = '';
			if (
				($item_ref->{ $iid }->{ amount } > 1) and 
				($item_ref->{ $iid }->{ gen_item_type } != 13) 	# Exclude coins. 
			) {
				$amountStr = " (" . $item_ref->{ $iid }->{ amount } . " pcs)";
			};
			&clientTell($cmqid, $username, $pid, "\t" . $item_ref->{ $iid }->{ shortDesc } . "$amountStr.\n");
		};
	};
	# Inform room: 
	if ($main::usersMainData[$uid]->{ invisible } == 0) {
		&informRoom($roomNum, $uid, "$username looks around.\n");
	};
};

sub cmdUnknown {
	my ($pid, $clientArgs, undef) = @_;
	my ($username, $uid, $cmqid, $roomNum, $gender) = &intPid2Details($pid);
	my $msg = "Unknown command: $clientArgs\n";
	&clientTell($cmqid, $username, $pid, $msg);
};

sub cmdSay {
	my ($pid, $clientArgs, undef) = @_;
	my ($username, $uid, $cmqid, $roomNum, $gender) = &intPid2Details($pid);
	&clientTell($cmqid, $username, $pid, "You say: \"$clientArgs\"\n");
	if ($main::usersMainData[$uid]->{ invisible }) {
		$username = 'Somebody';
	};
	&informRoom($roomNum, $uid, "$username says: \"$clientArgs\"\n");
};

sub cmdGo {
	# Move to direction:
	#my ($dir, $oldpos, $rest) = split(/ /, $cmdArgs, 3);
	my ($pid, $direction, undef) = @_;
	my ($username, $uid, $cmqid, $roomNum, $gender) = &intPid2Details($pid);
	$direction = uc($direction);
	$direction = quotemeta($direction);
	my $longDirection = '';

	if (scalar(grep(/^$direction$/, @main::longDirections)) == 1) {
		# It's long.
		$longDirection = $direction;
		$direction = $main::dirLtS{$direction};
	} elsif (scalar(grep(/^$direction$/, @main::directions)) == 1) {
		# It's short.
		$longDirection = $main::dirStL{$direction};
	} else {
		&clientTell($cmqid, $username, $pid, "Go where exactly?\n");
		if ($main::usersMainData[$uid]->{ invisible } == 0) {
			&informRoom($roomNum, $uid, "$username looks disoriented.\n");
		};
		return;
	};
	my $dirSTR = "ex$direction";
	my $newpos = $main::roomsData[$roomNum]{ $dirSTR };
	if ($newpos eq '') {
		&clientTell($cmqid, $username, $pid, "Go where exactly?\n");
		if ($main::usersMainData[$uid]->{ invisible } == 0) {
			&informRoom($roomNum, $uid, "$username looks disoriented.\n");
		};
		&logPrint("This should not happen: $username failed to go $dirSTR from rn $roomNum.\n");
		return;
	};
	if ($newpos == 0) {
		&clientTell($cmqid, $username, $pid, "Boing!\n");
		if ($main::usersMainData[$uid]->{ invisible } == 0) {
			if ($direction eq 'U') {
				&informRoom($roomNum, $uid, "$username jumps. Maybe $main::lex_he[$gender] tries to fly!\n");
			} elsif ($direction eq 'D') {
				&informRoom($roomNum, $uid, "$username smashes $main::lex_his[$gender] head on the floor!\n");
			} else {
				&informRoom($roomNum, $uid, "$username bumps in the " . lc($longDirection) . " wall!\n");
			};
		};
		return; 
	};
	if ($main::usersMainData[$uid]->{ invisible } == 0) {
		&informRoom($roomNum, $uid, "$username goes " . lc($longDirection) . ".\n");
	};
	# Move player:
	&movePlayer($uid, $newpos);
	# Inform player and new room!
	&clientTell($cmqid, $username, $pid, "You are in " . $main::roomsShortDescr[$newpos] . "\n");
	if ($main::usersMainData[$uid]->{ invisible } == 0) {
		&informRoom($newpos, $uid, "$username just came in.\n");
	};
	# Print room:
	#&clientPrintRoom($cmqid, $username, $pid);
	&cmdLook($pid);
	return; 
};

sub cmdTeleport {
	my ($pid, $args, undef) = @_;
	my ($newpos, $victim) = split(/ /, $args, 2);
	if (not defined $victim ) { $victim = '';};
	if (not defined $newpos ) { $newpos = -1;};
	my ($username, $uid, $cmqid, $roomNum, $gender) = &intPid2Details($pid);
	# Check if possible. For now, only gods can achieve teleport.
	# Later will add more.
	if (not $main::usersMainData[$uid]->{ godMode } == 1) {
		&clientTell($cmqid, $username, $pid, "Your magic failed!\n");
		if ($main::usersMainData[$uid]->{ invisible } == 0) {
			&informRoom($roomNum, $uid, "$username tries to cast a spell but fails.\n");
		};
		return;
	};
	# Define if it's self-teleport or other:
	# First, check if we need to teleport TO someone:
	if (uc($newpos) eq 'TO') {
		my $v_uid = -1;
		# Define victim:
		foreach my $lc_uid (keys @main::usersMainData) {
			if (not defined $main::usersMainData[$lc_uid]) {
				next;
			};
			if (&intCmpUid2name($lc_uid, $victim)) {
				$v_uid = $lc_uid;
				# In case of NPCs, more than 1 match is possible. Only first one
				# is matched.
			};
		};
		# Find position of target:
		if ($v_uid < 0) {
			&clientTell($cmqid, $username, $pid, "Your magic failed! $victim does not exists!\n");
			if ($main::usersMainData[$uid]->{ invisible } == 0) {
				&informRoom($roomNum, $uid, "$username tries to cast a spell but fails.\n");
			};
			return;
		};
		my $v_roomNum = $main::usersMainData[$v_uid]->{ room };
		if ($v_roomNum == $roomNum) {
			&clientTell($cmqid, $username, $pid, "Your magic failed! $victim is here!\n");
			if ($main::usersMainData[$uid]->{ invisible } == 0) {
				&informRoom($roomNum, $uid, "$username tries to cast a spell but fails.\n");
			};
			return;
		};
		# Otherwise, move:
		&clientTell($cmqid, $username, $pid, "You feel strange!\n");
		if ($main::usersMainData[$uid]->{ invisible } == 0) {
			&informRoom($roomNum, $uid, "$username disappears in a puf of smoke!\n");
		} else {
			&informRoom($roomNum, $uid, "There is a puf of smoke!\n");
		};
		&movePlayer($uid, $v_roomNum);
		if ($main::usersMainData[$uid]->{ invisible } == 0) {
			&informRoom($v_roomNum, $uid, "$username appears in a puf of smoke!\n");
		} else {
			&informRoom($v_roomNum, $uid, "There is a puf of smoke!\n");
		};
		&clientTell($cmqid, $username, $pid, "You are teleported in " . $main::roomsShortDescr[$v_roomNum] . "\n");
		return;
	};
	# Check if there is a username as a second argument:
	if ($victim ne '') {
		# Find target:
		my $tv_uid = -1;
		foreach my $lc_uid (keys @main::usersMainData) {
			if (not defined $main::usersMainData[$lc_uid]) {
				next;
			};
			if (&intCmpUid2name($lc_uid, $victim)) {
				$tv_uid = $lc_uid;
				# In case of NPCs, more than 1 match is possible. Only first one
				# is matched.
			};
		};
		if ($tv_uid < 0) {
			&clientTell($cmqid, $username, $pid, "Your magic failed! $victim does not exists!\n");
			if ($main::usersMainData[$uid]->{ invisible } == 0) {
				&informRoom($roomNum, $uid, "$username tries to cast a spell but fails.\n");
			};
			return;
		};
		if (not defined $main::roomsShortDescr[$newpos]) {
			&clientTell($cmqid, $username, $pid, "Your magic failed! You can't teleport $victim into the void!\n");
			if ($main::usersMainData[$uid]->{ invisible } == 0) {
				&informRoom($roomNum, $uid, "$username tries to cast a spell but fails.\n");
			};
			return;
		};
		# Get victim's details:
		my $v_pid = $main::usersMainData[$tv_uid]->{ pid };
		if ($v_pid == -1) {
			# Frozen user, fail.
			&clientTell($cmqid, $username, $pid, "Your magic failed! You can't teleport statues!\n");
			if ($main::usersMainData[$uid]->{ invisible } == 0) {
				&informRoom($roomNum, $uid, "$username tries to cast a spell but fails.\n");
			};
			return;
		};
		my ($v_username, $v_uid, $v_cmqid, $v_roomNum, $v_gender);
		if ($v_pid > 0) {
			# Real, logged in user
			($v_username, $v_uid, $v_cmqid, $v_roomNum, $v_gender) = &intPid2Details($v_pid);
		} else {
			# NPC. Find necessary data.
			$v_username = $main::usersMainData[$tv_uid]->{ username };
			$v_uid = $tv_uid;
			$v_cmqid = -2;
			$v_roomNum = $main::usersMainData[$tv_uid]->{ room };
		};
		# Inform caster's room:
		if ($main::usersMainData[$uid]->{ invisible } == 0) {
			&informRoom($roomNum, $uid, "$username casts a spell!\n");
		};
		# Teleport victim:
		&userTell($v_cmqid, $v_username, $v_pid, $v_uid, "You feel strange!\n");
		if ($main::usersMainData[$v_uid]->{ invisible } == 0) {
			&informRoom($v_roomNum, $v_uid, "$v_username disappears in a puf of smoke!\n");
		} else {
			&informRoom($v_roomNum, $v_uid, "There is a puf of smoke!\n");
		};
		&movePlayer($v_uid, $newpos);
		if ($main::usersMainData[$v_uid]->{ invisible } == 0) {
			&informRoom($newpos, $v_uid, "$v_username appears in a puf of smoke!\n");
		} else {
			&informRoom($newpos, $v_uid, "There is a puf of smoke!\n");
		};
		&userTell($v_cmqid, $v_username, $v_pid, $v_uid, "You are teleported in " . $main::roomsShortDescr[$newpos] . "\n");
		# Inform caster:
		&clientTell($cmqid, $username, $pid, "You teleported $v_username in " . $main::roomsShortDescr[$newpos] . "\n");
		return;
	};
	&clientTell($cmqid, $username, $pid, "You feel strange!\n");
	# Check if it is possible to teleport there. For now, 
	# checking only room existance.
	if ((not defined $main::roomsShortDescr[$newpos]) or ($newpos < 1)) {
		if ($main::usersMainData[$uid]->{ invisible } == 0) {
			&informRoom($roomNum, $uid, "$username disappears in a puf of smoke!\n");
			&informRoom($roomNum, $uid, "$username appears in a puf of smoke, looking terrified!\n");
		} else {
			&informRoom($roomNum, $uid, "There is a puf of smoke, and another one!\n");
		};
		&clientTell($cmqid, $username, $pid, "Wow! You make a fast pass from the beyond!\n");
		return;
	};
	# All ok, teleport.
	if ($main::usersMainData[$uid]->{ invisible } == 0) {
		&informRoom($roomNum, $uid, "$username disappears in a puf of smoke!\n");
	} else {
		&informRoom($roomNum, $uid, "There is a puf of smoke!\n");
	};
	&movePlayer($uid, $newpos);
	if ($main::usersMainData[$uid]->{ invisible } == 0) {
		&informRoom($newpos, $uid, "$username appears in a puf of smoke!\n");
	} else {
		&informRoom($newpos, $uid, "There is a puf of smoke!\n");
	};
	&clientTell($cmqid, $username, $pid, "You are teleported in " . $main::roomsShortDescr[$newpos] . "\n");
};

sub cmdStatus {
	# Print user details
	my ($pid, undef) = @_;
	my ($username, $uid, $cmqid, $roomNum, $gender) = &intPid2Details($pid);
	&clientTell($cmqid, $username, $pid, "$username the " . $main::userLevels{ $main::usersMainData[$uid]->{ level } } . ", " . $main::usersMainData[$uid]->{ desc } . ".\n");
	&clientTell($cmqid, $username, $pid, "HP: " . $main::usersMainData[$uid]->{ hitPoints } . "/" . $main::usersMainData[$uid]->{ maxHp } . "\t\tMana: " . $main::usersMainData[$uid]->{ mana } . "/" . $main::usersMainData[$uid]->{ maxMana } . "\n");
	&clientTell($cmqid, $username, $pid, "Coins: " . $main::usersMainData[$uid]->{ coins } . "\t\tGender: " . $main::lex_gender[$main::usersMainData[$uid]->{ gender }] . "\n");
	my $nextLevel = &intAdvExp($main::usersMainData[$uid]->{ level } + 1);
	&clientTell($cmqid, $username, $pid, "Room #: " . $main::usersMainData[$uid]->{ room } . "\t\tXP: " . $main::usersMainData[$uid]->{ xp } . " ($nextLevel)\n");
	&clientTell($cmqid, $username, $pid, "Energy: " . $main::usersMainData[$uid]->{ fuel } . "%\t\t " . '' . " \n");

	my @v_stat = ();
	&clientTell($cmqid, $username, $pid, "Last save: " . $main::usersMainData[$uid]->{ lastSave } . "\t\tLast Logout: " . $main::usersMainData[$uid]->{ lastDiscon } . "\n");
	&clientTell($cmqid, $username, $pid, "Last login: " . $main::usersMainData[$uid]->{ lastCon } . "\t\tE-mail: " . $main::usersMainData[$uid]->{ email } . "\n");
	&clientTell($cmqid, $username, $pid, "Alignement: " . &defAlignement($main::usersMainData[$uid]->{ alignement }) . "\n");
	$main::usersMainData[$uid]->{ godMode } && push(@v_stat, 'God like');
	$main::usersMainData[$uid]->{ paralyzed } && push(@v_stat, 'paralyzed');
	$main::usersMainData[$uid]->{ invisible } && push(@v_stat, 'invisible');
	$main::usersMainData[$uid]->{ invincible } && push(@v_stat, 'invincible');
	$main::usersMainData[$uid]->{ blind } && push(@v_stat, 'blind');
	$main::usersMainData[$uid]->{ intoxicated } && push(@v_stat, 'intoxicated');
	$main::usersMainData[$uid]->{ poisoned } && push(@v_stat, 'poisoned');
	my $v_stat = join(', ', @v_stat);
	if ($v_stat eq '') {$v_stat = 'None.';};
	&clientTell($cmqid, $username, $pid, "Attributes: $v_stat\n");
	&clientTell($cmqid, $username, $pid, 'Quests: ' . scalar(keys %{ $main::usersMainData[ $uid ]->{ quests } } ) . '/' . scalar(keys %main::questData) . "\n");
};

sub cmdHelp {
	my ($pid, $args) = @_;
	my ($username, $uid, $cmqid, $roomNum, $gender) = &intPid2Details($pid);
	$args =~ s/^(.*)\n//;
	$args = lc($args);
	if ($args eq '') { $args = 'main' };
	if (lc($args) eq 'list') {
		my $cnt = 0;
		# Special case
		&clientTell($cmqid, $username, $pid, "Command list:\n");
		&clientTell($cmqid, $username, $pid, "#\tLevel\tCommand\n");
		foreach my $cmd (sort { $a cmp $b } keys %$main::cmdhash_ref) {
			if ($main::usersMainData[$uid]->{ level } >= $main::cmdhash_ref->{ $cmd }->{ reqLevel }) {
				&clientTell($cmqid, $username, $pid, ++$cnt . "\t$main::cmdhash_ref->{ $cmd }->{ reqLevel }\t$cmd\n");
			};
		};
		return;
	};
	my $fn = $main::helpFileDir . $args . '.txt';
	open (HF, "<", $fn) or do { 
				&clientTell($cmqid, $username, $pid, "Help for \"$args\" not found.\n");
				return;
			};
	while (my $line = <HF>) {
		&clientTell($cmqid, $username, $pid, $line);
	};
};

sub cmdSet {
	my ($pid, $args) = @_;
	my ($username, $uid, $cmqid, $roomNum, $gender) = &intPid2Details($pid);
	my $subcmd;
	($subcmd, $args) = split(/ /, $args, 2);
	if (
		(not defined $subcmd) or 
		($subcmd eq '')
	) {
		&clientTell($cmqid, $username, $pid, "Syntax: SET DESCR|EMAIL|PASSWD|GENDER <argument>\n");
		return;

	};
	if (
		(not defined $args) or
		($args eq '')
	) {
		&clientTell($cmqid, $username, $pid, "This command needs an argument.\n");
		return;
	};
	if (uc($subcmd) eq 'DESCR') {
		$main::usersMainData[$uid]->{ desc } = $args;	
		&clientTell($cmqid, $username, $pid, "You look different!\n");
		if ($main::usersMainData[$uid]->{ invisible } == 0) {
			&informRoom($roomNum, $uid, "$username looks somehow different!\n");
		};
	} elsif (uc($subcmd) eq 'EMAIL') {
		$main::usersMainData[$uid]->{ email } = $args;	
		&clientTell($cmqid, $username, $pid, "E-mail changed.\n");
	} elsif (uc($subcmd) eq 'PASSWD') {
		$main::usersMainData[$uid]->{ password } = $args;	
		$main::userPass_ref->{$username}->{password} = $args;
		&clientTell($cmqid, $username, $pid, "Password changed to \'$args\'.\n");
	} elsif (uc($subcmd) eq 'GENDER') {
		my $gender;
		if (uc($args) eq 'MALE') {
			$gender = 0;
		} elsif (uc($args) eq 'FEMALE') {
			$gender = 1;
		} elsif (uc($args) eq 'OTHER') {
			$gender = 2;
		} else {
			&clientTell($cmqid, $username, $pid, "Please choose one of male, female, other.\n");
			return;
		};
		if ($main::usersMainData[$uid]->{ gender } == $gender) {
			&clientTell($cmqid, $username, $pid, "You already are $args.\n");
			return;
		};
		$main::usersMainData[$uid]->{ gender } = $gender;
		&clientTell($cmqid, $username, $pid, "You are now $args.\n");
		if ($main::usersMainData[$uid]->{ invisible } == 0) {
			&informRoom($roomNum, $uid, "$username looks VERY different!\n");
		};
	} else {
		&clientTell($cmqid, $username, $pid, "SET $subcmd not implemented.\n");
	};
	return;
};

sub cmdWho {
	# Connected player's list:
	my ($pid, $args) = @_;
	my ($username, $uid, $cmqid, $roomNum, $gender) = &intPid2Details($pid);
	foreach my $l_username (@main::loggedInUsers) {
		my $l_uid = $main::userPass_ref->{$l_username}->{uid};
		my $msg = "$l_username the " . $main::userLevels{ $main::usersMainData[$l_uid]->{ level } } . "\t\t(" . $main::usersMainData[$l_uid]->{ lastCon } . ")\n";
		&clientTell($cmqid, $username, $pid, $msg);
	};
};

sub cmdGet {
	my ($pid, $args) = @_;
	my ($username, $uid, $cmqid, $roomNum, $gender) = &intPid2Details($pid);
	my $item_ref = $main::roomsData[ $roomNum ]->{ presentItems };
	my @items = ();
	if ($args =~ /^#/) {
		# Special syntax with internal item id to resolve ambiguities
		my ($scrap, $iid) = split /#/, $args, 2;
		# if (not defined($item_ref->{ $iid }->{ invisible })) {
		if (not defined($item_ref->{ $iid })) {
			&clientTell($cmqid, $username, $pid, "Item #$iid invalid.\n");
			return;
		};
		if (
			($item_ref->{ $iid }->{ invisible } == 1) or 
			($item_ref->{ $iid }->{ movable } == 0) or
			($item_ref->{ $iid }->{ roomID } == 0) or
			($item_ref->{ $iid }->{ roomID } != $roomNum)
		) {
			&clientTell($cmqid, $username, $pid, "Item #$iid invalid.\n");
			return;
		};
		push @items, $iid;
	} else {
		# Search:
		$args = quotemeta($args);
		foreach my $iid (keys %$item_ref) {
			if ($item_ref->{ $iid }->{ invisible } == 0) { 	# Only visible and movable
				my $shortDesc = $item_ref->{ $iid }->{ shortDesc };
				if ($shortDesc =~ m/$args/i) {
					push @items, $iid;
				};
			};
		};
	};
	if (scalar(@items) == 0) {
		# Not found.
		&clientTell($cmqid, $username, $pid, "There is no $args here.\n");
	} elsif (scalar(@items) > 1) {
		# Found more than one:
		&clientTell($cmqid, $username, $pid, "Please be more specific. Get what of the following?\n");
		foreach my $iid (@items) {
			&clientTell($cmqid, $username, $pid, "#$iid\t: " . $item_ref->{ $iid }->{ shortDesc } . ".\n");
		};
	} else {
		# Exactly one: 
		my $iid = pop @items;
		if (not $main::itemData{ $iid }->{ movable }) {
			&clientTell($cmqid, $username, $pid, "You can't get that!\n");
			return;
		};
		# If item is multiple, decrease amount, create a new one and use the
		# new iid for the rest of the command.
		if ($main::itemData{ $iid }->{ amount } > 1) {
			my $niid = &intCreateItemFromItem($roomNum, $iid, 0);
			if (not $niid) {
				# Error:
				&clientTell($cmqid, $username, $pid, "You failed to get $item_ref->{ $iid }->{ shortDesc } (internal error).\n");
				return;
			};
			# New item created, decrease amount and work with new item:
			$main::itemData{ $iid }->{ amount } -= 1;
			$main::itemData{ $niid }->{ amount } = 1;
			$iid = $niid;
		};

		# Change main data structure:
		$main::itemData{ $iid }->{ roomID } = 0;
		$main::itemData{ $iid }->{ userID } = $uid;
		# Change room data structure:
		my $rec = $main::roomsData[ $roomNum ]->{ presentItems }->{ $iid };
		delete $main::roomsData[ $roomNum ]->{ presentItems }->{ $iid };
		# Change inventory data structure: 
		$rec->{ roomID } = 0;
		$rec->{ userID } = $uid;
		$main::usersMainData[ $uid ]->{ inventory }->{ $iid } = $rec;
		my $shortDesc = $main::usersMainData[ $uid ]->{ inventory }->{ $iid }->{ shortDesc };
		&clientTell($cmqid, $username, $pid, "You got $shortDesc.\n");
		if ($main::usersMainData[$uid]->{ invisible } == 0) {
			&informRoom($roomNum, $uid, "$username takes $shortDesc.\n");
		} else {
			&informRoom($roomNum, $uid, "Somebody takes $shortDesc.\n");
		};
		# Consolidate:
		&intConsolidateInv($main::usersMainData[ $uid ]->{ inventory });
	};
};

sub cmdDrop {
	my ($pid, $args) = @_;
	my ($username, $uid, $cmqid, $roomNum, $gender) = &intPid2Details($pid);
	my $inv_ref = $main::usersMainData[ $uid ]->{ inventory };
	my @items = ();
	my ($amount, $keyword) = split / /, $args, 2;
	if (not defined $amount) { $amount = ''; };
	if (not defined $keyword) { $keyword = ''; };
	$amount = quotemeta($amount);
	$keyword = quotemeta($keyword);
	if (
		(defined $keyword) and 
		('coins' =~ /^$keyword/i) and 
		(length($keyword) > 0) and 
		($amount =~ /^\d+$/)) {
		# Looks like user wants to drop some coins:
		if ($amount > 0) {
			if ($amount > $main::usersMainData[$uid]->{ coins }) {
				&clientTell($cmqid, $username, $pid, "You don\'t have this amount.\n");
				return;

			};
			my $rc = &intCreateMoney($roomNum, $amount);
			if ($rc) {
				$main::usersMainData[$uid]->{ coins } -= $amount;
				&clientTell($cmqid, $username, $pid, "You drop $amount coins.\n");
				if ($main::usersMainData[$uid]->{ invisible } == 0) {
					&informRoom($roomNum, $uid, "$username drops $amount coins.\n");
				} else {
					&informRoom($roomNum, $uid, "Somebody drops $amount coins.\n");
				};
				return;
			} else {
				&clientTell($cmqid, $username, $pid, "Sorry, something went wrong. You fail to drop $amount coins.\n");
				return;
			};
		} else {
			&clientTell($cmqid, $username, $pid, "Only whole numbers of coins greater than zero, please.\n");
			return;
		};
	};
	if ($args =~ /^#/) {
		# Special syntax with internal item id to resolve ambiguities
		my ($scrap, $iid) = split /#/, $args, 2;
		if (not defined $inv_ref->{ $iid }) {
			&clientTell($cmqid, $username, $pid, "Item #$iid invalid.\n");
			return;
		};
		if (
			($inv_ref->{ $iid }->{ dropable } == 0) or 
			($inv_ref->{ $iid }->{ userID } == 0) or
			($inv_ref->{ $iid }->{ userID } != $uid)
		) {
			&clientTell($cmqid, $username, $pid, "Item #$iid invalid.\n");
			return;
		};
		push @items, $iid;
	} else {
		my $args = quotemeta($args);
		foreach my $iid (keys %$inv_ref) {
			if ($inv_ref->{ $iid }->{ dropable } == 1) { 	# Only dropable
				my $shortDesc = $inv_ref->{ $iid }->{ shortDesc };
				if ($shortDesc =~ m/$args/i) {
					push @items, $iid;
				};
			};
		};
	};
	if (scalar(@items) == 0) {
		# Not found.
		&clientTell($cmqid, $username, $pid, "You have not any $args to drop.\n");
	} elsif (scalar(@items) > 1) {
		# Found more than one:
		&clientTell($cmqid, $username, $pid, "Please be more specific. Drop which one?\n");
		foreach my $iid (@items) {
			my $extra = '';
			if ($inv_ref->{ $iid }->{ worn }) {
				$extra = ' (worn)';
			};
			if ($inv_ref->{ $iid }->{ wielded }) {
				$extra = ' (wielded)';
			};
			&clientTell($cmqid, $username, $pid, "#$iid\t: " . $inv_ref->{ $iid }->{ shortDesc } . "$extra.\n");
		};
	} else {
		# Exactly one: 
		my $iid = pop @items;
		# Takeoff or unwield. No need to alter inventory record, as it will be deleted right after:
		my $shortDesc = $main::itemData{ $iid }->{ shortDesc };
		if ($main::itemData{ $iid }->{ worn } == 1) {
			$main::itemData{ $iid }->{ worn } = 0;
			&clientTell($cmqid, $username, $pid, "You take off $shortDesc.\n");
			&informRoom($roomNum, $uid, "$username takes off $shortDesc\n");
		};
		if ($main::itemData{ $iid }->{ wielded } == 1) {
			$main::itemData{ $iid }->{ wielded } = 0;
			&clientTell($cmqid, $username, $pid, "You unwield $shortDesc.\n");
			&informRoom($roomNum, $uid, "$username unwields $shortDesc\n");
		};
		# If item is multiple, decrease amount, create a new one and use the
		# new iid for the rest of the command.
		if ($main::itemData{ $iid }->{ amount } > 1) {
			my $niid = &intCreateItemFromItem($pid, $iid, 1);
			if (not $niid) {
				# Error:
				&clientTell($cmqid, $username, $pid, "You failed to drop $shortDesc (internal error).\n");
				return;
			};
			# New item created, decrease amount and work with new item:
			$main::itemData{ $iid }->{ amount } -= 1;
			$main::itemData{ $niid }->{ amount } = 1;
			$iid = $niid;
		};
		# Change main data structure:
		$main::itemData{ $iid }->{ roomID } = $roomNum;
		$main::itemData{ $iid }->{ userID } = 0;
		# Change inventory data structure: 
		my $rec = $main::usersMainData[ $uid ]->{ inventory }->{ $iid };
		delete $main::usersMainData[ $uid ]->{ inventory }->{ $iid };
		# Change room data structure:
		$rec->{ roomID } = $roomNum;
		$rec->{ userID } = 0;
		$main::roomsData[ $roomNum ]->{ presentItems }->{ $iid } = $rec;
		&clientTell($cmqid, $username, $pid, "You drop $shortDesc.\n");
		if ($main::usersMainData[$uid]->{ invisible } == 0) {
			&informRoom($roomNum, $uid, "$username drops $shortDesc.\n");
		} else {
			&informRoom($roomNum, $uid, "Somebody drops $shortDesc.\n");
		};
		# Consolidate
		&intConsolidateInv($main::roomsData[ $roomNum ]->{ presentItems });
	};
	return;
};

sub cmdInventory {
	# More to be done here, for now just a list.
	my ($pid, $args) = @_;
	my ($username, $uid, $cmqid, $roomNum, $gender) = &intPid2Details($pid);
	if (scalar(keys %{ $main::usersMainData[ $uid ]->{ inventory } }) == 0) {
		&clientTell($cmqid, $username, $pid, "You have nothing.\n");
	} else {
		# Consolidate:
		&intConsolidateInv($main::usersMainData[ $uid ]->{ inventory });

		&clientTell($cmqid, $username, $pid, "You carry:\n");
		foreach my $iid (keys %{ $main::usersMainData[ $uid ]->{ inventory } }) {
			my $shortDesc = $main::usersMainData[ $uid ]->{ inventory }->{ $iid }->{ shortDesc };
			my $amount = '';
			if ($main::usersMainData[ $uid ]->{ inventory }->{ $iid }->{ amount } > 1) {
				$amount = " (" . $main::usersMainData[ $uid ]->{ inventory }->{ $iid }->{ amount } . " pcs)";
			};
			my $worn = '';
			if ($main::usersMainData[ $uid ]->{ inventory }->{ $iid }->{ worn } == 1) {
				$worn = ' (worn)';
			};
			my $wielded = '';
			if ($main::usersMainData[ $uid ]->{ inventory }->{ $iid }->{ wielded } == 1) {
				$wielded = ' (wielded)';
			};
			&clientTell($cmqid, $username, $pid, "$shortDesc$amount$worn$wielded.\n");
		};
	};
};

sub cmdExamine {
	my ($pid, $args) = @_;
	my ($username, $uid, $cmqid, $roomNum, $gender) = &intPid2Details($pid);
	my ($item, $item2, $target) = split(/ /, $args, 3);
	if (not defined $item) { $item = '';};
	if (not defined $item2) { $item2 = '';};
	if (not defined $target) { $target = '';};
	my @items = ();	# Descr list in case of ambiguity.
	my @items_iid = ();	# iid list in case of ambiguity.
	my $found = ''; # Found in: R=room, I=inventory.
	my $fiid;	# On single, use this item id.
	my $v_username = '';
	my $v_uid;
	if ($item =~ /^#/) {
		# Special syntax with internal item id to resolve ambiguities
		my ($scrap, $iid) = split /#/, $item, 2;
		# Get item data from main data structure, it could be anywhere.
		my $single_item_ref;  
		if (defined($main::itemData{$iid})) {
			$single_item_ref = $main::itemData{$iid};
		} else {
			&clientTell($cmqid, $username, $pid, "Item #$iid invalid.\n");
			return;
		};

		if (
			($single_item_ref->{ invisible } == 1) or 
			(
				# Owned and owner not here.
				($single_item_ref->{ roomID } == 0) and
				($main::usersMainData[ $single_item_ref->{ userID } ]->{ room } != $roomNum)
			) or
			(
				# Laying in a different room
				($single_item_ref->{ roomID } != $roomNum) and
				($single_item_ref->{ userID } = 0)
			)
		) {
			&clientTell($cmqid, $username, $pid, "Item #$iid invalid.\n");
			return;
		};
		push @items_iid, $iid;
		$fiid = $iid;
		push @items, $single_item_ref->{ shortDesc };
		# Clear rest of arguments:
		$item2 = $target = '';
	} elsif (uc($item) eq 'MY') {
		# examine MY <item> syntax.
		# Only search inventory.
		if (scalar(keys %{ $main::usersMainData[ $uid ]->{ inventory } }) == 0) {
			&clientTell($cmqid, $username, $pid, "You have nothing to examine.\n");
			return;
		} else {
			# Search inventory:
			foreach my $iid (keys %{ $main::usersMainData[ $uid ]->{ inventory } }) {
				my $shortDesc = $main::usersMainData[ $uid ]->{ inventory }->{ $iid }->{ shortDesc };
				$item2 = quotemeta($item2);
				if ($shortDesc =~ m/$item2/i) {
					push @items, $shortDesc;
					push @items_iid, $iid;
					$found = 'I';
					$fiid = $iid;
				};
			};
		};
	} elsif (uc($item2) eq 'ON') {
		# examine <item> ON <person> syntax.
		# First check for person
		if (scalar(@{ $main::roomsData[ $roomNum ]->{presentUids} }) > 0) {
			foreach my $ouid (@{ $main::roomsData[ $roomNum ]->{presentUids} }) {
				# Loop through present persons and look for a username match:
				if (&intCmpUid2name($ouid, $target)) {
				#if (uc($main::usersMainData[$ouid]->{ username }) eq uc($target)) {
					$v_username = $main::usersMainData[$ouid]->{ username };
					$v_uid = $ouid;
					# Loop through user's inventory
					my $v_inv_ref = $main::usersMainData[$ouid]->{ inventory };
					$item = quotemeta($item);
					foreach my $viid (keys %$v_inv_ref) {
						if ((($v_inv_ref->{ $viid }->{ worn } == 1) or ($v_inv_ref->{ $viid }->{ wielded } == 1)) and ($v_inv_ref->{ $viid }->{ shortDesc } =~ m/$item/i)) {
							push @items, $v_inv_ref->{ $viid }->{ shortDesc };
							push @items_iid, $viid;
							$fiid = $viid;
						};
					};
				};
			};
			if ($v_username eq '') {
				# User not present, return now.
				&clientTell($cmqid, $username, $pid, "$target is not here.\n");
				return;
			};
		} else {
			# Nobody here.
			&clientTell($cmqid, $username, $pid, "$target is not here.\n");
			return;
		};
	} else {
		# Simple examine <item>
		if ((scalar(keys %{ $main::usersMainData[ $uid ]->{ inventory } }) == 0)
			and (scalar( keys %{ $main::roomsData[ $roomNum ]->{ presentItems } })) == 0) {
			&clientTell($cmqid, $username, $pid, "You have nothing to examine.\n");
			return;
		} else {
			# Search inventory:
			$item = quotemeta($item);
			foreach my $iid (keys %{ $main::usersMainData[ $uid ]->{ inventory } }) {
				my $shortDesc = $main::usersMainData[ $uid ]->{ inventory }->{ $iid }->{ shortDesc };
				if ($shortDesc =~ m/$item/i) {
					push @items, $shortDesc;
					push @items_iid, $iid;
					$found = 'I';
					$fiid = $iid;
				};
			};
			# Search room:
			my $item_ref = $main::roomsData[ $roomNum ]->{ presentItems };
			foreach my $iid (keys %$item_ref) {
				if ($item_ref->{ $iid }->{ invisible } == 0) { 	# Only visible
					my $shortDesc = $item_ref->{ $iid }->{ shortDesc };
					if ($shortDesc =~ m/$item/i) {
						push @items, $shortDesc;
						push @items_iid, $iid;
						$found = 'R';
						$fiid = $iid;
					};
				};
			};
		};
	};
	# Now check what we found:
	if (scalar(@items) == 0) {
		# Nothing found.
		if (uc($item) eq 'MY') { 
			&clientTell($cmqid, $username, $pid, "You don't have any $item2.\n");
		} elsif (uc($item2) eq 'ON') {
			&clientTell($cmqid, $username, $pid, "$v_username doesn't have any $target.\n");
		} else {;
			&clientTell($cmqid, $username, $pid, "No $item around.\n");
		};	
		return;
	} elsif (scalar(@items) > 1) {
		# More than one:
		&clientTell($cmqid, $username, $pid, "Please be more specific. Examine which one?\n");
		foreach my $iid (@items_iid) {
			my $extra = '';
			if ($main::itemData{ $iid }->{ worn }) {
				$extra = ' (worn)';
			};
			if ($main::itemData{ $iid }->{ wielded }) {
				$extra = ' (wielded)';
			};
			&clientTell($cmqid, $username, $pid, "#$iid\t:" . $main::itemData{ $iid }->{ shortDesc } . "$extra.\n");
		};
		return;
	} else {
		# Exactly one:
		my $item_ref = $main::itemData{ $fiid };
		my $formatted = autoformat $item_ref->{ longDesc }, { left=>1, right=>$main::termWidth, justify => 'full' };
		&clientTell($cmqid, $username, $pid, $formatted);
#		if ($item_ref->{ wearable } == 1) {
#			&clientTell($cmqid, $username, $pid, "Armour class: " . $item_ref->{ armourClass } . ".\n");
#		};
#		if ($item_ref->{ wieldable } == 1) {
#			&clientTell($cmqid, $username, $pid, "Weapon class: " . $item_ref->{ weaponClass } . ".\n");
#		};
		if (
			($item_ref->{ executable } ne '') and
			($item_ref->{ validCode } == 1)
		) {
			no strict "refs";
			# Load code:
			#my $code = "require '" . $item_ref->{ executable } . "'";
			#eval $code;
			#warn $@ if $@;
			&{ $item_ref->{ code_ref } }($pid, '__USE');
		};
		if (uc($item2) eq 'ON') {
			# Inform room and target.
			my $msg = "$username examines " . $main::usersMainData[$v_uid]->{ username } . "'s " . $item_ref->{ shortDesc } . ".\n";
			if ($main::usersMainData[$uid]->{ invisible } == 0) {
				&informRoom($roomNum, $uid, $msg, $v_uid);
				# Looked at:
				if ($main::usersMainData[$v_uid]->{ pid } > 0) {
					&clientTell(	$main::usersMainData[$v_uid]->{ qid },
							$main::usersMainData[$v_uid]->{ username },
							$main::usersMainData[$v_uid]->{ pid },
							"$username examines your " . $item_ref->{ shortDesc } . ".\n");
				} elsif ($main::usersMainData[$v_uid]->{ npc } == 1) {
					# Call interact routine of NPC, if available.
					&npcTell($pid, $v_uid, '__LOOKEDAT', '');
				};
			};
		};
	};
};

sub cmdWear {
	my ($pid, $args) = @_;
	&intWearTakeoff($pid, 'WE', $args);
};

sub cmdTakeoff {
	my ($pid, $args) = @_;
	&intWearTakeoff($pid, 'TO', $args);
};

sub cmdWield {
	my ($pid, $args) = @_;
	&intWieldUnwield($pid, 'WI', $args);
};

sub cmdUnwield {
	my ($pid, $args) = @_;
	&intWieldUnwield($pid, 'UN', $args);
};

sub cmdCreateItem {
	my ($pid, $args) = @_;
	my ($username, $uid, $cmqid, $roomNum, $gender) = &intPid2Details($pid);
	# Create a new item from a prototype. Argument is itemProto ID. 
	# Without arguments, a list of prototypes is returned.
	# TODO: The list can be too long, do something about that.
	# Update: Added second argument to create new item on a user other
	# than current.
	my $ouid;
	($args, $ouid) = split / /, $args, 2;
	if ((not defined $args) or ($args eq '')) { 
		foreach my $iid (keys %main::itemProto) {
			&clientTell($cmqid, $username, $pid, "$iid\t" . $main::itemProto{$iid}->{ shortDesc } . "\n");
		};
		return;
	} else {
		if (not defined $main::itemProto{$args}) {
			&clientTell($cmqid, $username, $pid, "Prototype $args not found.\n");
			return;
		};
	};
	if (
		(defined($ouid)) and
		(not defined $main::usersMainData[$ouid]->{ username })
	) {
		&clientTell($cmqid, $username, $pid, "UID $ouid not found.\n");
		return;
	};
	my $ok = &intCreateItemFromProto($pid, $args, 1);
	if ($ok) {
		# Inform everybody.
		&clientTell($cmqid, $username, $pid, "You cast an item creation spell.\n");
		if ($main::usersMainData[$uid]->{ invisible } == 0) {
			&informRoom($roomNum, $uid, "$username casts a spell!\n");
		};
		if (defined ($ouid)) {
			# Change main data structure:
			$main::itemData{ $ok }->{ userID } = $ouid;
			# Change inventory data structure: 
			# Remove from donor:
			my $rec = $main::usersMainData[ $uid ]->{ inventory }->{ $ok };
			delete $main::usersMainData[ $uid ]->{ inventory }->{ $ok };
			# Add to recipient:
			$rec->{ userID } = $ouid;
			$main::usersMainData[ $ouid ]->{ inventory }->{ $ok } = $rec;
			&clientTell($cmqid, $username, $pid, $main::itemData{ $ok }->{ shortDesc } . " (#$ok) added to $main::usersMainData[$ouid]->{ username } (#$ouid)\n");
		} else {
			&clientTell($cmqid, $username, $pid, $main::itemData{ $ok }->{ shortDesc } . " (#$ok) appears in your hands!\n");
			if ($main::usersMainData[$uid]->{ invisible } == 0) {
				&informRoom($roomNum, $uid, $main::itemData{ $ok }->{ shortDesc } . " appears in $username\'s hands!\n");
			};
		};

	} else {
		# Only inform user about failure.
		&clientTell($cmqid, $username, $pid, "Failed due to lock.\n");
	};
};

sub cmdDestroyItem {
	my ($pid, $args) = @_;
	my ($username, $uid, $cmqid, $roomNum, $gender) = &intPid2Details($pid);
	my @items = ();
	my @items_iid = ();
	my $found = '';
	my $fiid;
	if ($args =~ /^#/) {
		# Special syntax with internal item id to resolve ambiguities
		my ($scrap, $iid) = split /#/, $args, 2;
		# Get item data from main data structure, it could be anywhere.
		my $single_item_ref;  
		if (defined($main::itemData{$iid})) {
			$single_item_ref = $main::itemData{$iid};
		} else {
			&clientTell($cmqid, $username, $pid, "Item #$iid invalid.\n");
			return;
		};
		if (
			($single_item_ref->{ invisible } == 1) or 
			(
				# Owned and owner not user.
				($single_item_ref->{ roomID } == 0) and
				($single_item_ref->{ userID } != $uid) 
			) or
			(
				# Laying in a different room
				($single_item_ref->{ roomID } != $roomNum) and
				($single_item_ref->{ userID } == 0)
			)
		) {
			&clientTell($cmqid, $username, $pid, "Item #$iid invalid.\n");
			return;
		};
		push @items_iid, $iid;
		$fiid = $iid;
		push @items, $single_item_ref->{ shortDesc };
	} elsif (
		(scalar(keys %{ $main::usersMainData[ $uid ]->{ inventory } }) == 0) and 
		(scalar( keys %{ $main::roomsData[ $roomNum ]->{ presentItems } }) == 0)
	) {
		&clientTell($cmqid, $username, $pid, "You have nothing to destroy.\n");
		return;
	} else {
		# Search inventory:
		if ($args ne  quotemeta($args)) {
			&clientTell($cmqid, $username, $pid, "Item invalid.\n");
			return;
		};
		foreach my $iid (keys %{ $main::usersMainData[ $uid ]->{ inventory } }) {
			my $shortDesc = $main::usersMainData[ $uid ]->{ inventory }->{ $iid }->{ shortDesc };
			if ($shortDesc =~ m/$args/i) {
				push @items, $shortDesc;
				push @items_iid, $iid;
				$found = 'I';
				$fiid = $iid;
			};
		};
		# Search room:
		my $item_ref = $main::roomsData[ $roomNum ]->{ presentItems };
		if ($args ne  quotemeta($args)) {
			&clientTell($cmqid, $username, $pid, "Item invalid.\n");
			return;
		};
		foreach my $iid (keys %$item_ref) {
			if ($item_ref->{ $iid }->{ invisible } == 0) { 	# Only visible
				my $shortDesc = $item_ref->{ $iid }->{ shortDesc };
				if ($shortDesc =~ m/$args/i) {
					push @items, $shortDesc;
					push @items_iid, $iid;
					$found = 'R';
					$fiid = $iid;
				};
			};
		};
	};
	# Now check what we found:
	if (scalar(@items) == 0) {
		# Nothing found.
		&clientTell($cmqid, $username, $pid, "No $args around.\n");
		return;
	} elsif (scalar(@items) > 1) {
		# More than one:
		&clientTell($cmqid, $username, $pid, "Please be more specific. Destroy which one?\n");
		foreach my $iid (@items_iid) {
			my $extra = '';
			if ($main::itemData{ $iid }->{ worn }) {
				$extra = ' (worn)';
			};
			if ($main::itemData{ $iid }->{ wielded }) {
				$extra = ' (wielded)';
			};
			&clientTell($cmqid, $username, $pid, "#$iid\t:" . $main::itemData{$iid}->{ shortDesc } . "$extra.\n");
		};
		return;
	} else {
		# Exactly one:
		my $ok = &intDestroyItem($fiid);
		# Inform room and user.
		my $shortDesc = pop @items;
		&clientTell($cmqid, $username, $pid, "You cast a spell on $shortDesc.\n");
		if ($main::usersMainData[$uid]->{ invisible } == 0) {
			&informRoom($roomNum, $uid, "$username casts a spell.\n");
		};
		&informRoom($roomNum, -1, "$shortDesc disappears in a bright flash of light!\n");
		return;
	};
};

sub cmdGive {
	my ($pid, $args) = @_;
	my ($username, $uid, $cmqid, $roomNum, $gender) = &intPid2Details($pid);
	my ($item, $victim) = split / to /i, $args, 2;
	if ((not defined $item) or (not defined $victim) or ($item eq '') or ($victim eq '')) {
		&clientTell($cmqid, $username, $pid, "Please be more specific.\n");
		return;
	};
	# Look for recipient:
	if (uc($username) eq uc($victim)) {
		&clientTell($cmqid, $username, $pid, "YOU are $username!\n");
		return;
	};
	if (scalar(keys %{ $main::usersMainData[ $uid ]->{ inventory } }) == 0) {
		&clientTell($cmqid, $username, $pid, "You have nothing to give.\n");
	};
	if (scalar(@{ $main::roomsData[ $roomNum ]->{presentUids} }) > 0)
		{
		# Persons.
		foreach my $ouid (@{ $main::roomsData[ $roomNum ]->{presentUids} }) {
			# Loop through present persons and look for a username match:
			#if (uc($main::usersMainData[$ouid]->{ username }) eq uc($victim)) {
			if (&intCmpUid2name($ouid, $victim)) {
				# Found user.
				if ($main::usersMainData[$ouid]->{ pid } == -1) {
					&clientTell($cmqid, $username, $pid, "You can't give anything to an ice statue.\n");
					return;
				};
				# Check for coins.
				my ($amount, $keyword) = split / /, $item, 2;
				if (not defined $amount) { $amount = ''; };
				if (not defined $keyword) { $keyword = ''; };
				if (
					($amount ne quotemeta($amount)) or
					($keyword ne quotemeta($keyword))
				) {
					&clientTell($cmqid, $username, $pid, "Invalid arguments.\n");
					return;
				};
				if (
					(defined $keyword) and 
					('coins' =~ /^$keyword/i) and 
					(length($keyword) > 0) and
					($amount =~ /^\d+$/)) {
					# Looks like user wants to give some coins:
					if ($amount > 0) {
						if ($amount > $main::usersMainData[ $uid ]->{ coins } ) {
							&clientTell($cmqid, $username, $pid, "You don\'t have this amount.\n");
							return;
						};
						$main::usersMainData[ $uid ]->{ coins } -= $amount;
						$main::usersMainData[ $ouid ]->{ coins } += $amount;
						# Inform
						my ($v_username, $v_pid, $v_cmqid, $v_roomNum, $v_gender) = &intUid2Details($ouid);
						&clientTell($cmqid, $username, $pid, "You give $amount coins to $v_username.\n");
						my $tt_username = 'Somebody';
						if ($main::usersMainData[$uid]->{ invisible } == 0) {
							$tt_username = $username;
						}; 
							
						&informRoom($roomNum, $uid, "$tt_username gives some coins to $v_username.\n", $ouid);
						if ($main::usersMainData[$ouid]->{ pid } > 0) {
							# Real person
							&clientTell($v_cmqid, $v_username, $v_pid, "$tt_username gives you $amount coins.\n");
						} elsif ($main::usersMainData[$ouid]->{ npc } == 1) {
							# Call interact routine of NPC, if available.
							&npcTell($pid, $ouid, '__GIVEN', "COINS$amount");
						};
						return;
					} else {
						&clientTell($cmqid, $username, $pid, "Only whole numbers of coins greater than zero, please.\n");
					};
				};

				# Check for item:
				my @items = ();
				my @items_iid = ();
				my $fiid;
				my $shortDesc;
				my $inv_ref = $main::usersMainData[ $uid ]->{ inventory };
				if ($item =~ /^#/) {
					# Special syntax with internal item id to resolve ambiguities
					my ($scrap, $iid) = split /#/, $item, 2;
					if (not defined($inv_ref->{ $iid })) {
						&clientTell($cmqid, $username, $pid, "Item #$iid invalid.\n");
						return;
					};
					if (
						($inv_ref->{ $iid }->{ dropable } == 0) or 
						($inv_ref->{ $iid }->{ userID } == 0) or
						($inv_ref->{ $iid }->{ userID } != $uid)
					) {
						&clientTell($cmqid, $username, $pid, "Item #$iid invalid.\n");
						return;
					};
					push @items_iid, $iid;
					$fiid = $iid;
					push @items, $inv_ref->{ $iid }->{ shortDesc };
				} else {
					$item = quotemeta($item);
					foreach my $iid (keys %{ $main::usersMainData[ $uid ]->{ inventory } }) {
						$shortDesc = $main::usersMainData[ $uid ]->{ inventory }->{ $iid }->{ shortDesc };
						if ($shortDesc =~ m/$item/i) {
							push @items, $shortDesc;
							push @items_iid, $iid;
							$fiid = $iid;
						};
					};
				};
				if (scalar(@items) == 0) {
					# Nothing found.
					&clientTell($cmqid, $username, $pid, "You have no $item.\n");
					return;
				} elsif (scalar(@items) > 1) {
					# More than one:
					&clientTell($cmqid, $username, $pid, "Please be more specific. Give which one?\n");
					foreach my $iid (@items_iid) {
						my $extra = '';
						if ($main::itemData{ $iid }->{ worn }) {
							$extra = ' (worn)';
						};
						if ($main::itemData{ $iid }->{ wielded }) {
							$extra = ' (wielded)';
						};
						&clientTell($cmqid, $username, $pid, "#$iid\t:" . $main::itemData{$iid}->{ shortDesc } . "$extra.\n");
					};
					return;
				} else {
					# Exactly one:
					$shortDesc = $main::usersMainData[ $uid ]->{ inventory }->{ $fiid }->{ shortDesc };
					if ($main::itemData{ $fiid }->{ worn } == 1) {
						$main::itemData{ $fiid }->{ worn } = 0;
						$main::usersMainData[ $uid ]->{ inventory }->{ $fiid}->{ worn } = 0;
						&clientTell($cmqid, $username, $pid, "You take off $shortDesc.\n");
						if ($main::usersMainData[$uid]->{ invisible } == 0) {
							&informRoom($roomNum, $uid, "$username takes off $shortDesc\n");
						};
					};
					if ($main::itemData{ $fiid }->{ wielded } == 1) {
						$main::itemData{ $fiid }->{ wielded } = 0;
						$main::usersMainData[ $uid ]->{ inventory }->{ $fiid }->{ wielded } = 0;
						&clientTell($cmqid, $username, $pid, "You unwield $shortDesc.\n");
						if ($main::usersMainData[$uid]->{ invisible } == 0) {
							&informRoom($roomNum, $uid, "$username unwields $shortDesc\n");
						};
					};

					# If item is multiple, decrease amount, create a new one and use the
					# new iid for the rest of the command.
					if ($main::itemData{ $fiid }->{ amount } > 1) {
						my $niid = &intCreateItemFromItem($pid, $fiid, 1);
						if (not $niid) {
							# Error:
							&clientTell($cmqid, $username, $pid, "You failed to give $shortDesc (internal error).\n");
							return;
						};
						# New item created, decrease amount and work with new item:
						$main::itemData{ $fiid }->{ amount } -= 1;
						$main::itemData{ $niid }->{ amount } = 1;
						$fiid = $niid;
					};

					# Change main data structure:
					$main::itemData{ $fiid }->{ userID } = $ouid;
					# Change inventory data structure: 
					# Remove from donor:
					my $rec = $main::usersMainData[ $uid ]->{ inventory }->{ $fiid };
					delete $main::usersMainData[ $uid ]->{ inventory }->{ $fiid };
					# Add to recipient:
					$rec->{ userID } = $ouid;
					$main::usersMainData[ $ouid ]->{ inventory }->{ $fiid } = $rec;
					# Inform
					my $tt_username = 'Somebody';
					if ($main::usersMainData[$uid]->{ invisible } == 0) {
						$tt_username = $username;
					};
					my ($v_username, $v_pid, $v_cmqid, $v_roomNum, $v_gender) = &intUid2Details($ouid);
					&clientTell($cmqid, $username, $pid, "You give $shortDesc to $v_username.\n");
					&informRoom($roomNum, $uid, "$tt_username gives $shortDesc to $v_username.\n", $ouid);
					if ($main::usersMainData[$ouid]->{ pid } > 0) {
						# Real person
						&clientTell($v_cmqid, $v_username, $v_pid, "$tt_username gives you $shortDesc.\n");
					} elsif ($main::usersMainData[$ouid]->{ npc } == 1) {
						# Call interact routine of NPC, if available.
						&npcTell($pid, $ouid, '__GIVEN', $fiid);
					};
					return;
				};
			};
		};
		&clientTell($cmqid, $username, $pid, "$victim is not here.\n");
		return;
	};
};

sub cmdShout {
	my ($pid, $msg, undef) = @_;
	my ($username, $uid, $cmqid, $roomNum, $gender) = &intPid2Details($pid);
	&clientTell($cmqid, $username, $pid, "You shout: \"$msg\"\n");
	$msg = "$username shouts: \"$msg\"\n";
	foreach my $t_username (@main::loggedInUsers) {
		&clientTell($main::usersMainData[ $main::userPass_ref->{$t_username}->{uid} ]->{ qid }, $t_username, $main::usersMainData[ $main::userPass_ref->{$t_username}->{uid} ]->{ pid }, $msg) if ($username ne $t_username);
	};
};

sub cmdHit {
	# This is more an 'attack' command. Will use both strings to 
	# call this one.
	my ($pid, $args) = @_;
	my ($username, $uid, $cmqid, $roomNum, $gender) = &intPid2Details($pid);
	if (not defined $args) { $args = ''; };
	if ($args eq '') {
		&clientTell($cmqid, $username, $pid, "This command needs an argument.\n");
		return;
	};
	if (uc($username) eq uc($args)) {
		&clientTell($cmqid, $username, $pid, "You are trying to hit yourself, but you miss!\n");
		if ($main::usersMainData[$uid]->{ invisible } == 0) {
			&informRoom($roomNum, $uid, "$username tries to hit " . $main::lex_his[$gender] . "self, but " . $main::lex_he[$gender] . " misses!\n");
		};
		return;
	};
	if ($main::usersMainData[$uid]->{ invincible } == 1) {
		&clientTell($cmqid, $username, $pid, "Your peacefull nature prevents you from attacking.\n");
		return;
	};
	if (scalar(@{ $main::roomsData[ $roomNum ]->{presentUids} }) > 0) {
		foreach my $ouid (@{ $main::roomsData[ $roomNum ]->{presentUids} }) {
			#if (uc($main::usersMainData[$ouid]->{ username }) eq uc($args)) {
			if (&intCmpUid2name($ouid, $args)) {
				if ($main::usersMainData[$ouid]->{ pid } == -1) {
					# Frozen.
					&clientTell($cmqid, $username, $pid, "You smash your hand on the frozen body of " . $main::usersMainData[$ouid]->{ username } . "!\n");
					if ($main::usersMainData[$uid]->{ invisible } == 0) {
						&informRoom($roomNum, $uid, "$username smashes " . $main::lex_his[$gender] . " hand on the frozen body of " . $main::usersMainData[$ouid]->{ username } . "!\n");
					};
					return;
				};
				if ($main::usersMainData[$ouid]->{ invincible } == 1) {
					&clientTell($cmqid, $username, $pid, $main::usersMainData[$ouid]->{ username } . " looks invincible! You decide not to attack.\n");
					return;
				};

				&createFight($uid, $ouid);
				return;
			};
		};
		&clientTell($cmqid, $username, $pid, "$args is not here.\n");
	} else {
		&clientTell($cmqid, $username, $pid, "You are alone.\n");
		return;
	};
	
};

sub cmdSummon {
	my ($pid, $args) = @_;
	my ($username, $uid, $cmqid, $roomNum, $gender) = &intPid2Details($pid);
	# Summon a new NPC from a prototype. Argument is npcProto ID. 
	# Without arguments, a list of prototypes is returned.
	# TODO: The list can be too long, do something about that.
	if ((not defined $args) or ($args eq '')) { 
		foreach my $luid (keys @main::npcProto) {
			if (not defined $main::npcProto[$luid]->{ npc }) {next;};
			&clientTell($cmqid, $username, $pid, "$luid\t" . $main::npcProto[$luid]{ username } . "\n");
		};
		return;
	} else {
		if (not defined $main::npcProto[$args]) {
			&clientTell($cmqid, $username, $pid, "NPC Prototype $args not found.\n");
			return;
		};
	};
	my $ok = &intCreateNpcFromProto($pid, $args, 1);
	if ($ok) {
		# Inform everybody.
		&clientTell($cmqid, $username, $pid, "You cast a summoning spell.\n");
		if ($main::usersMainData[$uid]->{ invisible } == 0) {
			&informRoom($roomNum, $uid, "$username casts a spell!\n");
		};
		&informRoom($roomNum, -1, $main::usersMainData[ $ok ]->{ username } . " appears from nowhere!\n");
		# Save new NPC
		&saveNewUser($ok);
	} else {
		# Only inform user about failure.
		&clientTell($cmqid, $username, $pid, "NPC summon Failed!\n");
	};
};

sub cmdWhisper {
	# Tell something to a player or NPC.
	my ($pid, $args) = @_;
	my ($username, $uid, $cmqid, $roomNum, $gender) = &intPid2Details($pid);
	my ($target, $message) = split / /, $args, 2;
	&intTell($pid, 'WHISPER', $target, $message);
};

sub cmdTell {
	# Tell something to a player or NPC. Similar to whisper, but message 
	# is passed to the room.
	my ($pid, $args) = @_;
	my ($username, $uid, $cmqid, $roomNum, $gender) = &intPid2Details($pid);
	my ($target, $message) = split / /, $args, 2;
	&intTell($pid, 'TELL', $target, $message);
};

sub cmdKiss {
	my ($pid, $args) = @_;
	&intDoOther($pid, 'KISS', $args);
};

sub cmdHug {
	my ($pid, $args) = @_;
	&intDoOther($pid, 'HUG', $args);
};

sub cmdHand {
	my ($pid, $args) = @_;
	&intDoOther($pid, 'HAND', $args);
};

sub cmdShake {
	my ($pid, $args) = @_;
	&intDoOther($pid, 'SHAKE', $args);
};

sub cmdSpit {
	my ($pid, $args) = @_;
	&intDoOther($pid, 'SPIT', $args);
};

sub cmdBite {
	my ($pid, $args) = @_;
	&intDoOther($pid, 'BITE', $args);
};

sub cmdKick {
	my ($pid, $args) = @_;
	&intDoOther($pid, 'KICK', $args);
};

sub cmdPush {
	my ($pid, $args) = @_;
	&intDoOther($pid, 'PUSH', $args);
};

sub cmdCaress {
	my ($pid, $args) = @_;
	&intDoOther($pid, 'CARESS', $args);
};

sub cmdTouch {
	my ($pid, $args) = @_;
	&intDoOther($pid, 'TOUCH', $args);
};

sub cmdBow {
	my ($pid, $args) = @_;
	my ($username, $uid, $cmqid, $roomNum, $gender) = &intPid2Details($pid);
	my $msg;
	if ((not defined $args) or ($args eq '')) {
		if ($main::usersMainData[$uid]->{ invisible } == 0) {
			$msg = "$username bows gracefully.\n";
			&informRoom($roomNum, $uid, $msg);
		};
		$msg = "You bow.\n";
		&clientTell($cmqid, $username, $pid, $msg);
	} else {
		&intDoOther($pid, 'BOW', $args);
	};
};

sub cmdSmile {
	my ($pid, $args) = @_;
	my ($username, $uid, $cmqid, $roomNum, $gender) = &intPid2Details($pid);
	if (not defined $args) {$args = '';};
	if ($main::usersMainData[$uid]->{ invisible }) {
		my $osmi = '';
		my $smile = '';
		if ($args eq 'strange') {
			$smile .= ' strangelly';
			$osmi .= ' strange';
		} elsif ($args =~ /evil/i) {
			$smile .= ' evilly';
			$osmi .= 'n evil';
		} elsif ($args =~ /sad/i) {
			$smile .= ' sadly';
			$osmi .= ' sad';
		} elsif ($args =~ /happy/i) {
			$smile .= ' happily';
			$osmi .= ' happy';
		} else {
		};
		my $msg = "A$osmi smile appears floating in the air!\n";
		&informRoom($roomNum, $uid, $msg);
		$msg = "You smile$smile.\n";
		&clientTell($cmqid, $username, $pid, $msg);
		return;
	};
	my $smile = ' smiles';
	my $osmi = '';
	if ($args eq 'strange') {
		$smile .= ' strangelly';
		$osmi .= ' strangelly';
	} elsif ($args =~ /evil/i) {
		$smile .= ' evilly';
		$osmi .= ' evilly';
	} elsif ($args =~ /sad/i) {
		$smile .= ' sadly';
		$osmi .= ' sadly';
	} elsif ($args =~ /happy/i) {
		$smile .= ' happily';
		$osmi .= ' happily';
	} else {
	};
	my $msg = "$username$smile.\n";
	&informRoom($roomNum, $uid, $msg);
	$msg = "You smile$osmi.\n";
	&clientTell($cmqid, $username, $pid, $msg);
};

sub cmdBlush {
	my ($pid, $args) = @_;
	my ($username, $uid, $cmqid, $roomNum, $gender) = &intPid2Details($pid);
	if ($main::usersMainData[$uid]->{ invisible }) {
		&informRoom($roomNum, $uid, "A red tint appears in the room\n");
		my $msg = "You blush.\n";
		&clientTell($cmqid, $username, $pid, $msg);
		return;
	};
	my $msg = "$username blushes.\n";
	&informRoom($roomNum, $uid, $msg);
	$msg = "You blush.\n";
	&clientTell($cmqid, $username, $pid, $msg);
};

sub cmdLaugh {
	my ($pid, $args) = @_;
	my ($username, $uid, $cmqid, $roomNum, $gender) = &intPid2Details($pid);
	my $msg;
	if ($main::usersMainData[$uid]->{ invisible }) {
		$msg = "You hear somebody falling down laughing!\n";
	} else {
		$msg = "$username falls down laughing!\n";
	};
	&informRoom($roomNum, $uid, $msg);
	$msg = "You fall down laughing.\n";
	&clientTell($cmqid, $username, $pid, $msg);
};

sub cmdGrin {
	my ($pid, $args) = @_;
	my ($username, $uid, $cmqid, $roomNum, $gender) = &intPid2Details($pid);
	my $msg;
	if ($main::usersMainData[$uid]->{ invisible }) {
		$msg = "A grin appears floating in the air.\n";
	} else {
		$msg = "$username grins.\n";
	};
	&informRoom($roomNum, $uid, $msg);
	$msg = "You grin.\n";
	&clientTell($cmqid, $username, $pid, $msg);
};

sub cmdQuests {
	my ($pid, $args) = @_;
	my ($username, $uid, $cmqid, $roomNum, $gender) = &intPid2Details($pid);
	my %slv = ();
    	while (my ($key, $value) = each %{ $main::usersMainData[ $uid ]->{ quests } }) {
    		$slv{$value} = $key;
    	};
	if ((defined $args) and ($args ne '')) {
		# Check for quest number
		if (defined $main::questData{$args}) {
			# Specific quest, print long description:
			my $msg = "Q. #$args: \t". $main::questData{$args}->{ shortDesc };
			if (defined $slv{$args}) {
				# Solved by user:
				$msg = "S$msg (solved)\n";
			} else {
				$msg = " $msg\n";
			};
			&clientTell($cmqid, $username, $pid, $msg);
			&clientPrintLong($cmqid, $username, $pid, $main::questData{$args}->{ longDesc });
			return;
		};
	};
	foreach my $qid (keys %main::questData) {
		my $msg = "Q. #$qid: \t". $main::questData{$qid}->{ shortDesc };
		if (defined $slv{$qid}) {
			# Solved by user:
			$msg = "S$msg (solved)\n";
		} else {
			$msg = " $msg\n";
		};
		&clientTell($cmqid, $username, $pid, $msg);
	};
};

sub cmdShoot {
	my ($pid, $direction, undef) = @_;
	my ($username, $uid, $cmqid, $roomNum, $gender) = &intPid2Details($pid);
	# Check for bow and arrow(s):
	my $inv_ref = $main::usersMainData[$uid]->{ inventory };
	my $res = my $mtype = my $ltype = my $miid = my $liid = 0;
	foreach my $iid (keys %$inv_ref) {
		if (($inv_ref->{ $iid }->{ wielded } == 1) and ($inv_ref->{ $iid }->{ missile } == 1)) {
			$mtype = $inv_ref->{ $iid }->{ missile_type };
			$miid = $iid;
			$res += 1;
		};
		if (($inv_ref->{ $iid }->{ wielded } == 1) and ($inv_ref->{ $iid }->{ missile_launcher } == 1)) {
			$ltype = $inv_ref->{ $iid }->{ missile_type_needed };
			$liid = $iid;
			$res += 2;
		};
	};
	if ($ltype == $mtype) { $res += 4; };
	if (($res & 2) == 0) {
		# No "bow"
		&clientTell($cmqid, $username, $pid, "You must wield a launcher item (like a bow) to shoot.\n");
		return 0;
	} elsif (($res & 1) == 0) {
		# No "arrow"
		&clientTell($cmqid, $username, $pid, "You must wield a missile item (like an arrow) to shoot.\n");
		return 0;
	} elsif (($res & 4) == 0) {
		# Mismatch 
		my $msg = $inv_ref->{ $liid }->{ shortDesc } . " will not work with " . $inv_ref->{ $miid }->{ shortDesc } . "!\n";
		&clientTell($cmqid, $username, $pid, $msg);
		return 0;
	} else {
		# Nothing.
	};
	# User has everything needed, now shoot:
	my $hitFactor = $inv_ref->{ $liid }->{ weaponClass };
	my $arrowFactor = $inv_ref->{ $miid }->{ weaponClass };
	my $hitOk = 0;
	# First iteration is special, as arrow is in the same room with shooter:
	$direction = uc($direction);
	my $longDirection = '';
	$direction = quotemeta($direction);
	if (scalar(grep(/^$direction$/, @main::longDirections)) == 1) {
		# It's long.
		$longDirection = $direction;
		$direction = $main::dirLtS{$direction};
	} elsif (scalar(grep(/^$direction$/, @main::directions)) == 1) {
		# It's short.
		$longDirection = $main::dirStL{$direction};
	} else {
		&clientTell($cmqid, $username, $pid, "Shoot where exactly?\n");
		return;
	};
	&clientTell($cmqid, $username, $pid, "You shoot " . $inv_ref->{ $miid }->{ shortDesc } . " " . lc($longDirection) . ".\n");
	if ($main::usersMainData[$uid]->{ invisible }) {
		&informRoom($roomNum, $uid, "Somebody shoots " . $inv_ref->{ $miid }->{ shortDesc } . " " . lc($longDirection) . ".\n");
	} else {
		&informRoom($roomNum, $uid, "$username shoots " . $inv_ref->{ $miid }->{ shortDesc } . " " . lc($longDirection) . ".\n");
	};
	my $dirSTR = "ex$direction";
	my $newpos = $main::roomsData[$roomNum]{ $dirSTR };
	if ($newpos == 0) {
		# Shoot in the wall:
		&informRoom($roomNum, -2, $inv_ref->{ $miid }->{ shortDesc } . " hits the stone and breaks.\n");
		# Remove one missile from shooter. 
		$inv_ref->{ $miid }->{ amount } -= 1;
		if ($inv_ref->{ $miid }->{ amount } < 1) {
			# Delete item:
			my $ok = &intDestroyItem($miid);
		};
		return;
	};
	# Now call recursive routine for the rest of range:
	$hitOk = &intMissileMove($roomNum, $newpos, $hitFactor, $arrowFactor, $miid, $dirSTR);
	if ($hitOk) {
		&clientTell($cmqid, $username, $pid, "You hit something!\n");
	} else {
		&clientTell($cmqid, $username, $pid, "You missed.\n");
	};
	# Remove one missile from shooter. 
	$inv_ref->{ $miid }->{ amount } -= 1;
	if ($inv_ref->{ $miid }->{ amount } < 1) {
		# Delete item:
		my $ok = &intDestroyItem($miid);
	};
};

sub cmdAdvance {
	my ($pid, undef) = @_;
	my ($username, $uid, $cmqid, $roomNum, $gender) = &intPid2Details($pid);
	my $level = $main::usersMainData[ $uid ]->{ level }; 
	my $xp = $main::usersMainData[ $uid ]->{ xp }; 
	my $nextxp = &intAdvExp($level+1);
	my $xpdiff = $nextxp - $xp;
	my $questNum = scalar(keys %{ $main::usersMainData[ $uid ]->{ quests } } );
	my $questTot = scalar(keys %main::questData);
	if ($level > 17) {
		&clientTell($cmqid, $username, $pid, "No more levels!\n");
		return;
	};
	if ($nextxp > $xp) {
		&clientTell($cmqid, $username, $pid, "Your level is $level. You have not enough experience to go to next level.\n");
		&clientTell($cmqid, $username, $pid, "You need $xpdiff for next level.\n");
		return;
	};
	if (
		($level == 17) and
		($questNum < $questTot)
	) {
		&clientTell($cmqid, $username, $pid, "You must solve " . $questTot . " quests to become a Mage.\n");
		return;
	};
	if ($level > 17) {
		&clientTell($cmqid, $username, $pid, "You are in the highest level.\n");
		return;
	};
	# All ok, advance level:
	$level++;
	my $newHp = $level*100;
	my $newMana = $level*100;
	$main::usersMainData[ $uid ]->{ level } = $level; 
	$main::usersMainData[ $uid ]->{ maxHp } = $newHp; 
	$main::usersMainData[ $uid ]->{ maxMana } = $newMana; 
	&clientTell($cmqid, $username, $pid, "Your level is now $level (" . $main::userLevels{ $main::usersMainData[$uid]->{ level } } . ").\n");
	if ($level == 18) {
		&tellAllUsers("A new Mage named $username is born!");
		&tellAllUsers("Great balls of fire explode in the sky!");
	};
	return;
};

sub cmdEat {
	my ($pid, $args) = @_;
	&intEatDrink('EAT', $pid, $args);
};

sub cmdDrink {
	my ($pid, $args) = @_;
	&intEatDrink('DRINK', $pid, $args);
};

sub intEatDrink {
	my ($subcmd, $pid, $args) = @_;
	my ($username, $uid, $cmqid, $roomNum, $gender) = &intPid2Details($pid);
	my @items = ();
	my @items_iid = ();
	my $found = '';
	my $fiid;
	my $EATtxt = 'eat';
	my $EATStxt = 'eats';
	if ($subcmd eq 'DRINK') {
		$EATtxt = 'drink';
		$EATStxt = 'drinks';
	};
	if ($args =~ /^#/) {
		# Special syntax with internal item id to resolve ambiguities
		my ($scrap, $iid) = split /#/, $args, 2;
		# Get item data from main data structure, it could be anywhere.
		my $single_item_ref;  
		if (defined($main::itemData{$iid})) {
			$single_item_ref = $main::itemData{$iid};
		} else {
			&clientTell($cmqid, $username, $pid, "Item #$iid invalid.\n");
			return;
		};
		if (
			(
				# Owned and owner not user.
				($single_item_ref->{ roomID } == 0) and
				($single_item_ref->{ userID } != $uid) 
			) or
			(
				# Laying in a different room
				($single_item_ref->{ roomID } != $roomNum) and
				($single_item_ref->{ userID } == 0)
			)
		) {
			&clientTell($cmqid, $username, $pid, "Item #$iid invalid.\n");
			return;
		};
		push @items_iid, $iid;
		$fiid = $iid;
		push @items, $single_item_ref->{ shortDesc };
	} elsif (scalar(keys %{ $main::usersMainData[ $uid ]->{ inventory } }) == 0) {
		&clientTell($cmqid, $username, $pid, "You have nothing to $EATtxt.\n");
		return;
	} else {
		# Search inventory:
		$args = quotemeta($args);
		foreach my $iid (keys %{ $main::usersMainData[ $uid ]->{ inventory } }) {
			my $shortDesc = $main::usersMainData[ $uid ]->{ inventory }->{ $iid }->{ shortDesc };
			if ($shortDesc =~ m/$args/i) {
				push @items, $shortDesc;
				push @items_iid, $iid;
				$found = 'I';
				$fiid = $iid;
			};
		};
	};
	# Now check what we found:
	if (scalar(@items) == 0) {
		# Nothing found.
		&clientTell($cmqid, $username, $pid, "You have no $args.\n");
		return;
	} elsif (scalar(@items) > 1) {
		# More than one:
		&clientTell($cmqid, $username, $pid, "Please be more specific, $EATtxt which one?\n");
		foreach my $iid (@items_iid) {
			&clientTell($cmqid, $username, $pid, "#$iid\t:" . $main::itemData{$iid}->{ shortDesc } . "\n");
		};
		return;
	} else {
		# Exactly one:
		# Inform room and user.
		my $shortDesc = pop @items;
		# Check item's attributes:
		if (
			((not $main::itemData{ $fiid }->{ edible }) and ($subcmd eq 'EAT')) or 
			((not $main::itemData{ $fiid }->{ drinkable }) and ($subcmd eq 'DRINK')) 
		) {
			&clientTell($cmqid, $username, $pid, "Silly! You can\'t $EATtxt $shortDesc.\n");
			&informRoom($roomNum, $uid, "$username suddently realizes that he can\'t $EATtxt $shortDesc.\n");
			return;
		};
		&clientTell($cmqid, $username, $pid, "You $EATtxt $shortDesc.\n");
		if ($main::usersMainData[$uid]->{ invisible }) {
			&informRoom($roomNum, $uid, "Somebody $EATStxt $shortDesc.\n");
		} else {
			&informRoom($roomNum, $uid, "$username $EATStxt $shortDesc.\n");
		};
		# Fuel:
		if ($main::itemData{ $fiid }->{ edible } > 1) {
			$main::usersMainData[$uid]->{ fuel } += $main::itemData{ $fiid }->{ edible };
			&clientTell($cmqid, $username, $pid, "Mmmm... Delicious...!\n");
			if ($main::usersMainData[$uid]->{ fuel } >= 100) {
				$main::usersMainData[$uid]->{ fuel } = 100;
				&clientTell($cmqid, $username, $pid, "You are full!\n");
			};
			if ($main::usersMainData[$uid]->{ invisible } == 0) {
				&informRoom($roomNum, $uid, "$username looks less hungry now.\n");
			};
		};	
		if ($main::itemData{ $fiid }->{ drinkable } > 1) {
			$main::usersMainData[$uid]->{ fuel } += $main::itemData{ $fiid }->{ drinkable };
			&clientTell($cmqid, $username, $pid, "Mmmm... Delicious...!\n");
			if ($main::usersMainData[$uid]->{ fuel } >= 100) {
				$main::usersMainData[$uid]->{ fuel } = 100;
				&clientTell($cmqid, $username, $pid, "You are full!\n");
			};
			if ($main::usersMainData[$uid]->{ invisible } == 0) {
				&informRoom($roomNum, $uid, "$username looks less hungry now.\n");
			};
		};	
		# Healing: 
		if ($main::itemData{ $fiid }->{ healing }) {
			$main::usersMainData[$uid]->{ hitPoints } += $main::itemData{ $fiid }->{ healing };
			if ($main::usersMainData[$uid]->{ hitPoints } >= $main::usersMainData[$uid]->{ maxHp }) {
				$main::usersMainData[$uid]->{ hitPoints } = $main::usersMainData[$uid]->{ maxHp };
			};
			&clientTell($cmqid, $username, $pid, "You feel healed!\n");
			if ($main::usersMainData[$uid]->{ invisible } == 0) {
				&informRoom($roomNum, $uid, "$username looks better now.\n");
			};
		};
		# Poisoned:
		if ($main::itemData{ $fiid }->{ poison }) {
			$main::usersMainData[$uid]->{ hitPoints } -= $main::itemData{ $fiid }->{ poison };
			if ($main::usersMainData[$uid]->{ hitPoints } <= 0 ) {
				$main::usersMainData[$uid]->{ hitPoints } = 1;
			};
			&clientTell($cmqid, $username, $pid, "Shit! It was poisoned! Aaarghh!\n");
			if ($main::usersMainData[$uid]->{ invisible } == 0) {
				&informRoom($roomNum, $uid, "$username screams from pain! Obviously, $shortDesc was poisoned!\n");
			} else {
				&informRoom($roomNum, $uid, "Somebody screams from pain! Obviously, $shortDesc was poisoned!\n");
			};
		};
		# Intoxicating:
		if ($main::itemData{ $fiid }->{ intox }) {
			$main::usersMainData[$uid]->{ intoxicated } += $main::itemData{ $fiid }->{ intox };
			if ($main::usersMainData[$uid]->{ intoxicated } >= $main::usersMainData[$uid]->{ maxHp } / 2 ) {
				$main::usersMainData[$uid]->{ intoxicated } = $main::usersMainData[$uid]->{ maxHp } / 2;
			};
			if ($main::itemData{ $fiid }->{ intox } > 250) {
				&clientTell($cmqid, $username, $pid, "You feel your guts burning!\n");
				if ($main::usersMainData[$uid]->{ invisible } == 0) {
					&informRoom($roomNum, $uid, "$username looks really drunk.\n");
				};
			} elsif ($main::itemData{ $fiid }->{ intox } > 150) {
				&clientTell($cmqid, $username, $pid, "A warm shock runs through your body.\n");
				if ($main::usersMainData[$uid]->{ invisible } == 0) {
					&informRoom($roomNum, $uid, "$username turns slightly red.\n");
				};
			} elsif ($main::itemData{ $fiid }->{ intox } > 30) {
				&clientTell($cmqid, $username, $pid, "Ah, that felt quite nice!\n");
				if ($main::usersMainData[$uid]->{ invisible } == 0) {
					&informRoom($roomNum, $uid, "$username turns slightly red.\n");
				};
			} else {
				&clientTell($cmqid, $username, $pid, "You burp.\n");
				if ($main::usersMainData[$uid]->{ invisible } == 0) {
					&informRoom($roomNum, $uid, "$username burps.\n");
				} else {
					&informRoom($roomNum, $uid, "Somebody burps.\n");
				};
			};
		};
		my $ok = &intDestroyItem($fiid);
		return;
	};
};

END { } # module clean-up code here (global destructor)

1; # don't forget to return a true value from the file
